/*****************************************************************************
 Copyright (c) 2017-2018, Intel Corporation

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:

     * Redistributions of source code must retain the above copyright notice,
       this list of conditions and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
     * Neither the name of Intel Corporation nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*****************************************************************************/

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <intel-ipsec-mb.h>
#include "gcm_ctr_vectors_test.h"
#include "utils.h"

#define BYTE_ROUND_UP(x) ((x + 7) / 8)
/*
 * Test Vector from
 * https://tools.ietf.org/html/rfc3686
 *
 */
/*
   Test Vector #1: Encrypting 16 octets using AES-CTR with 128-bit key
   AES Key          : AE 68 52 F8 12 10 67 CC 4B F7 A5 76 55 77 F3 9E
   AES-CTR IV       : 00 00 00 00 00 00 00 00
   Nonce            : 00 00 00 30
   Plaintext String : 'Single block msg'
   Plaintext        : 53 69 6E 67 6C 65 20 62 6C 6F 63 6B 20 6D 73 67
   Counter Block (1): 00 00 00 30 00 00 00 00 00 00 00 00 00 00 00 01
   Key Stream    (1): B7 60 33 28 DB C2 93 1B 41 0E 16 C8 06 7E 62 DF
   Ciphertext       : E4 09 5D 4F B7 A7 B3 79 2D 61 75 A3 26 13 11 B8
*/
static const uint8_t K1_CTR[] = {
        0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
        0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E,
};
static const uint8_t IV1_CTR[] = {
        0x00, 0x00, 0x00, 0x30,	/* nonce */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};
static const uint8_t P1_CTR[] = {
        0x53, 0x69, 0x6E, 0x67, 0x6C, 0x65, 0x20, 0x62,
        0x6C, 0x6F, 0x63, 0x6B, 0x20, 0x6D, 0x73, 0x67,
};
static const uint8_t C1_CTR[] = {
        0xE4, 0x09, 0x5D, 0x4F, 0xB7, 0xA7, 0xB3, 0x79,
        0x2D, 0x61, 0x75, 0xA3, 0x26, 0x13, 0x11, 0xB8,
};
static const uint8_t T1_CTR[] = { 0 };
static const uint8_t A1_CTR[] = { 0 };
#define A1_CTR_len 0

/*
   Test Vector #2: Encrypting 32 octets using AES-CTR with 128-bit key
   AES Key          : 7E 24 06 78 17 FA E0 D7 43 D6 CE 1F 32 53 91 63
   AES-CTR IV       : C0 54 3B 59 DA 48 D9 0B
   Nonce            : 00 6C B6 DB
   Plaintext        : 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F
                    : 10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F
   Counter Block (1): 00 6C B6 DB C0 54 3B 59 DA 48 D9 0B 00 00 00 01
   Key Stream    (1): 51 05 A3 05 12 8F 74 DE 71 04 4B E5 82 D7 DD 87
   Counter Block (2): 00 6C B6 DB C0 54 3B 59 DA 48 D9 0B 00 00 00 02
   Key Stream    (2): FB 3F 0C EF 52 CF 41 DF E4 FF 2A C4 8D 5C A0 37
   Ciphertext       : 51 04 A1 06 16 8A 72 D9 79 0D 41 EE 8E DA D3 88
                    : EB 2E 1E FC 46 DA 57 C8 FC E6 30 DF 91 41 BE 28
*/
static const uint8_t K2_CTR[] = {
        0x7E, 0x24, 0x06, 0x78, 0x17, 0xFA, 0xE0, 0xD7,
        0x43, 0xD6, 0xCE, 0x1F, 0x32, 0x53, 0x91, 0x63,
};
static const uint8_t IV2_CTR[] = {
        0x00, 0x6C, 0xB6, 0xDB,	/* nonce */
        0xC0, 0x54, 0x3B, 0x59, 0xDA, 0x48, 0xD9, 0x0B,
};
static const uint8_t P2_CTR[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
};
static const uint8_t C2_CTR[] = {
        0x51, 0x04, 0xA1, 0x06, 0x16, 0x8A, 0x72, 0xD9,
        0x79, 0x0D, 0x41, 0xEE, 0x8E, 0xDA, 0xD3, 0x88,
        0xEB, 0x2E, 0x1E, 0xFC, 0x46, 0xDA, 0x57, 0xC8,
        0xFC, 0xE6, 0x30, 0xDF, 0x91, 0x41, 0xBE, 0x28,
};
static const uint8_t T2_CTR[] = { 0 };
static const uint8_t A2_CTR[] = { 0 };
#define A2_CTR_len 0

/*
   Test Vector #3: Encrypting 36 octets using AES-CTR with 128-bit key
   AES Key          : 76 91 BE 03 5E 50 20 A8 AC 6E 61 85 29 F9 A0 DC
   AES-CTR IV       : 27 77 7F 3F  4A 17 86 F0
   Nonce            : 00 E0 01 7B
   Plaintext        : 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F
                    : 10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F
                    : 20 21 22 23
   Counter Block (1): 00 E0 01 7B 27 77 7F 3F 4A 17 86 F0 00 00 00 01
   Key Stream    (1): C1 CE 4A AB 9B 2A FB DE C7 4F 58 E2 E3 D6 7C D8
   Counter Block (2): 00 E0 01 7B 27 77 7F 3F 4A 17 86 F0 00 00 00 02
   Key Stream    (2): 55 51 B6 38 CA 78 6E 21 CD 83 46 F1 B2 EE 0E 4C
   Counter Block (3): 00 E0 01 7B 27 77 7F 3F 4A 17 86 F0 00 00 00 03
   Key Stream    (3): 05 93 25 0C 17 55 36 00 A6 3D FE CF 56 23 87 E9
   Ciphertext       : C1 CF 48 A8 9F 2F FD D9 CF 46 52 E9 EF DB 72 D7
                    : 45 40 A4 2B DE 6D 78 36 D5 9A 5C EA AE F3 10 53
                    : 25 B2 07 2F
*/
static const uint8_t K3_CTR[] = {
        0x76, 0x91, 0xBE, 0x03, 0x5E, 0x50, 0x20, 0xA8,
        0xAC, 0x6E, 0x61, 0x85, 0x29, 0xF9, 0xA0, 0xDC,
};
static const uint8_t IV3_CTR[] = {
        0x00, 0xE0, 0x01, 0x7B,	/* nonce */
        0x27, 0x77, 0x7F, 0x3F, 0x4A, 0x17, 0x86, 0xF0,
};
static const uint8_t P3_CTR[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
        0x20, 0x21, 0x22, 0x23,
};
static const uint8_t C3_CTR[] = {
        0xC1, 0xCF, 0x48, 0xA8, 0x9F, 0x2F, 0xFD, 0xD9,
        0xCF, 0x46, 0x52, 0xE9, 0xEF, 0xDB, 0x72, 0xD7,
        0x45, 0x40, 0xA4, 0x2B, 0xDE, 0x6D, 0x78, 0x36,
        0xD5, 0x9A, 0x5C, 0xEA, 0xAE, 0xF3, 0x10, 0x53,
        0x25, 0xB2, 0x07, 0x2F,
};
static const uint8_t T3_CTR[] = { 0 };
static const uint8_t A3_CTR[] = { 0 };
#define A3_CTR_len 0

/*
   Test Vector #4: Encrypting 16 octets using AES-CTR with 192-bit key
   AES Key          : 16 AF 5B 14 5F C9 F5 79 C1 75 F9 3E 3B FB 0E ED
                    : 86 3D 06 CC FD B7 85 15
   AES-CTR IV       : 36 73 3C 14 7D 6D 93 CB
   Nonce            : 00 00 00 48
   Plaintext String : 'Single block msg'
   Plaintext        : 53 69 6E 67 6C 65 20 62 6C 6F 63 6B 20 6D 73 67
   Counter Block (1): 00 00 00 48 36 73 3C 14 7D 6D 93 CB 00 00 00 01
   Key Stream    (1): 18 3C 56 28 8E 3C E9 AA 22 16 56 CB 23 A6 9A 4F
   Ciphertext       : 4B 55 38 4F E2 59 C9 C8 4E 79 35 A0 03 CB E9 28
*/
static const uint8_t K4_CTR[] = {
        0x16, 0xAF, 0x5B, 0x14, 0x5F, 0xC9, 0xF5, 0x79,
        0xC1, 0x75, 0xF9, 0x3E, 0x3B, 0xFB, 0x0E, 0xED,
        0x86, 0x3D, 0x06, 0xCC, 0xFD, 0xB7, 0x85, 0x15,
};
static const uint8_t IV4_CTR[] = {
        0x00, 0x00, 0x00, 0x48,	/* nonce */
        0x36, 0x73, 0x3C, 0x14, 0x7D, 0x6D, 0x93, 0xCB,
};
static const uint8_t P4_CTR[] = {
        0x53, 0x69, 0x6E, 0x67, 0x6C, 0x65, 0x20, 0x62,
        0x6C, 0x6F, 0x63, 0x6B, 0x20, 0x6D, 0x73, 0x67,
};
static const uint8_t C4_CTR[] = {
        0x4B, 0x55, 0x38, 0x4F, 0xE2, 0x59, 0xC9, 0xC8,
        0x4E, 0x79, 0x35, 0xA0, 0x03, 0xCB, 0xE9, 0x28,
};
static const uint8_t T4_CTR[] = { 0 };
static const uint8_t A4_CTR[] = { 0 };
#define A4_CTR_len 0

/*
   Test Vector #5: Encrypting 32 octets using AES-CTR with 192-bit key
   AES Key          : 7C 5C B2 40 1B 3D C3 3C 19 E7 34 08 19 E0 F6 9C
                    : 67 8C 3D B8 E6 F6 A9 1A
   AES-CTR IV       : 02 0C 6E AD C2 CB 50 0D
   Nonce            : 00 96 B0 3B
   Plaintext        : 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F
                    : 10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F
   Counter Block (1): 00 96 B0 3B 02 0C 6E AD C2 CB 50 0D 00 00 00 01
   Key Stream    (1): 45 33 41 FF 64 9E 25 35 76 D6 A0 F1 7D 3C C3 90
   Counter Block (2): 00 96 B0 3B 02 0C 6E AD C2 CB 50 0D 00 00 00 02
   Key Stream    (2): 94 81 62 0F 4E C1 B1 8B E4 06 FA E4 5E E9 E5 1F
   Ciphertext       : 45 32 43 FC 60 9B 23 32 7E DF AA FA 71 31 CD 9F
                    : 84 90 70 1C 5A D4 A7 9C FC 1F E0 FF 42 F4 FB 00
*/
static const uint8_t K5_CTR[] = {
        0x7C, 0x5C, 0xB2, 0x40, 0x1B, 0x3D, 0xC3, 0x3C,
        0x19, 0xE7, 0x34, 0x08, 0x19, 0xE0, 0xF6, 0x9C,
        0x67, 0x8C, 0x3D, 0xB8, 0xE6, 0xF6, 0xA9, 0x1A,
};
static const uint8_t IV5_CTR[] = {
        0x00, 0x96, 0xB0, 0x3B,	/* nonce */
        0x02, 0x0C, 0x6E, 0xAD, 0xC2, 0xCB, 0x50, 0x0D,
};
static const uint8_t P5_CTR[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
};
static const uint8_t C5_CTR[] = {
        0x45, 0x32, 0x43, 0xFC, 0x60, 0x9B, 0x23, 0x32,
        0x7E, 0xDF, 0xAA, 0xFA, 0x71, 0x31, 0xCD, 0x9F,
        0x84, 0x90, 0x70, 0x1C, 0x5A, 0xD4, 0xA7, 0x9C,
        0xFC, 0x1F, 0xE0, 0xFF, 0x42, 0xF4, 0xFB, 0x00,
};
static const uint8_t T5_CTR[] = { 0 };
static const uint8_t A5_CTR[] = { 0 };
#define A5_CTR_len 0

/*
   Test Vector #6: Encrypting 36 octets using AES-CTR with 192-bit key
   AES Key          : 02 BF 39 1E E8 EC B1 59 B9 59 61 7B 09 65 27 9B
                    : F5 9B 60 A7 86 D3 E0 FE
   AES-CTR IV       : 5C BD 60 27 8D CC 09 12
   Nonce            : 00 07 BD FD
   Plaintext        : 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F
                    : 10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F
                    : 20 21 22 23
   Counter Block (1): 00 07 BD FD 5C BD 60 27 8D CC 09 12 00 00 00 01
   Key Stream    (1): 96 88 3D C6 5A 59 74 28 5C 02 77 DA D1 FA E9 57
   Counter Block (2): 00 07 BD FD 5C BD 60 27 8D CC 09 12 00 00 00 02
   Key Stream    (2): C2 99 AE 86 D2 84 73 9F 5D 2F D2 0A 7A 32 3F 97
   Counter Block (3): 00 07 BD FD 5C BD 60 27 8D CC 09 12 00 00 00 03
   Key Stream    (3): 8B CF 2B 16 39 99 B2 26 15 B4 9C D4 FE 57 39 98
   Ciphertext       : 96 89 3F C5 5E 5C 72 2F 54 0B 7D D1 DD F7 E7 58
                    : D2 88 BC 95 C6 91 65 88 45 36 C8 11 66 2F 21 88
                    : AB EE 09 35
*/
static const uint8_t K6_CTR[] = {
        0x02, 0xBF, 0x39, 0x1E, 0xE8, 0xEC, 0xB1, 0x59,
        0xB9, 0x59, 0x61, 0x7B, 0x09, 0x65, 0x27, 0x9B,
        0xF5, 0x9B, 0x60, 0xA7, 0x86, 0xD3, 0xE0, 0xFE,
};
static const uint8_t IV6_CTR[] = {
        0x00, 0x07, 0xBD, 0xFD,	/* nonce */
        0x5C, 0xBD, 0x60, 0x27, 0x8D, 0xCC, 0x09, 0x12,
};
static const uint8_t P6_CTR[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
        0x20, 0x21, 0x22, 0x23,
};
static const uint8_t C6_CTR[] = {
        0x96, 0x89, 0x3F, 0xC5, 0x5E, 0x5C, 0x72, 0x2F,
        0x54, 0x0B, 0x7D, 0xD1, 0xDD, 0xF7, 0xE7, 0x58,
        0xD2, 0x88, 0xBC, 0x95, 0xC6, 0x91, 0x65, 0x88,
        0x45, 0x36, 0xC8, 0x11, 0x66, 0x2F, 0x21, 0x88,
        0xAB, 0xEE, 0x09, 0x35,
};
static const uint8_t T6_CTR[] = { 0 };
static const uint8_t A6_CTR[] = { 0 };
#define A6_CTR_len 0

/*
   Test Vector #7: Encrypting 16 octets using AES-CTR with 256-bit key
   AES Key          : 77 6B EF F2 85 1D B0 6F 4C 8A 05 42 C8 69 6F 6C
                    : 6A 81 AF 1E EC 96 B4 D3 7F C1 D6 89 E6 C1 C1 04
   AES-CTR IV       : DB 56 72 C9 7A A8 F0 B2
   Nonce            : 00 00 00 60
   Plaintext String : 'Single block msg'
   Plaintext        : 53 69 6E 67 6C 65 20 62 6C 6F 63 6B 20 6D 73 67
   Counter Block (1): 00 00 00 60 DB 56 72 C9 7A A8 F0 B2 00 00 00 01
   Key Stream    (1): 47 33 BE 7A D3 E7 6E A5 3A 67 00 B7 51 8E 93 A7
   Ciphertext       : 14 5A D0 1D BF 82 4E C7 56 08 63 DC 71 E3 E0 C0
*/
static const uint8_t K7_CTR[] = {
        0x77, 0x6B, 0xEF, 0xF2, 0x85, 0x1D, 0xB0, 0x6F,
        0x4C, 0x8A, 0x05, 0x42, 0xC8, 0x69, 0x6F, 0x6C,
        0x6A, 0x81, 0xAF, 0x1E, 0xEC, 0x96, 0xB4, 0xD3,
        0x7F, 0xC1, 0xD6, 0x89, 0xE6, 0xC1, 0xC1, 0x04,
};
static const uint8_t IV7_CTR[] = {
        0x00, 0x00, 0x00, 0x60,	/* nonce */
        0xDB, 0x56, 0x72, 0xC9, 0x7A, 0xA8, 0xF0, 0xB2,
};
static const uint8_t P7_CTR[] = {
        0x53, 0x69, 0x6E, 0x67, 0x6C, 0x65, 0x20, 0x62,
        0x6C, 0x6F, 0x63, 0x6B, 0x20, 0x6D, 0x73, 0x67,
};
static const uint8_t C7_CTR[] = {
        0x14, 0x5A, 0xD0, 0x1D, 0xBF, 0x82, 0x4E, 0xC7,
        0x56, 0x08, 0x63, 0xDC, 0x71, 0xE3, 0xE0, 0xC0,
};
static const uint8_t T7_CTR[] = { 0 };
static const uint8_t A7_CTR[] = { 0 };
#define A7_CTR_len 0

/*
   Test Vector #8: Encrypting 32 octets using AES-CTR with 256-bit key
   AES Key          : F6 D6 6D 6B D5 2D 59 BB 07 96 36 58 79 EF F8 86
                    : C6 6D D5 1A 5B 6A 99 74 4B 50 59 0C 87 A2 38 84
   AES-CTR IV       : C1 58 5E F1 5A 43 D8 75
   Nonce            : 00 FA AC 24
   Plaintext        : 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F
                    : 10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F
   Counter block (1): 00 FA AC 24 C1 58 5E F1 5A 43 D8 75 00 00 00 01
   Key stream    (1): F0 5F 21 18 3C 91 67 2B 41 E7 0A 00 8C 43 BC A6
   Counter block (2): 00 FA AC 24 C1 58 5E F1 5A 43 D8 75 00 00 00 02
   Key stream    (2): A8 21 79 43 9B 96 8B 7D 4D 29 99 06 8F 59 B1 03
   Ciphertext       : F0 5E 23 1B 38 94 61 2C 49 EE 00 0B 80 4E B2 A9
                    : B8 30 6B 50 8F 83 9D 6A 55 30 83 1D 93 44 AF 1C
*/
static const uint8_t K8_CTR[] = {
        0xF6, 0xD6, 0x6D, 0x6B, 0xD5, 0x2D, 0x59, 0xBB,
        0x07, 0x96, 0x36, 0x58, 0x79, 0xEF, 0xF8, 0x86,
        0xC6, 0x6D, 0xD5, 0x1A, 0x5B, 0x6A, 0x99, 0x74,
        0x4B, 0x50, 0x59, 0x0C, 0x87, 0xA2, 0x38, 0x84,
};
static const uint8_t IV8_CTR[] = {
        0x00, 0xFA, 0xAC, 0x24,	/* nonce */
        0xC1, 0x58, 0x5E, 0xF1, 0x5A, 0x43, 0xD8, 0x75,
};
static const uint8_t P8_CTR[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
};
static const uint8_t C8_CTR[] = {
        0xF0, 0x5E, 0x23, 0x1B, 0x38, 0x94, 0x61, 0x2C,
        0x49, 0xEE, 0x00, 0x0B, 0x80, 0x4E, 0xB2, 0xA9,
        0xB8, 0x30, 0x6B, 0x50, 0x8F, 0x83, 0x9D, 0x6A,
        0x55, 0x30, 0x83, 0x1D, 0x93, 0x44, 0xAF, 0x1C,
};
static const uint8_t T8_CTR[] = { 0 };
static const uint8_t A8_CTR[] = { 0 };
#define A8_CTR_len 0

/*
   Test Vector #9: Encrypting 36 octets using AES-CTR with 256-bit key
   AES Key          : FF 7A 61 7C E6 91 48 E4 F1 72 6E 2F 43 58 1D E2
                    : AA 62 D9 F8 05 53 2E DF F1 EE D6 87 FB 54 15 3D
   AES-CTR IV       : 51 A5 1D 70 A1 C1 11 48
   Nonce            : 00 1C C5 B7
   Plaintext        : 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F
                    : 10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F
                    : 20 21 22 23
   Counter block (1): 00 1C C5 B7 51 A5 1D 70 A1 C1 11 48 00 00 00 01
   Key stream    (1): EB 6D 50 81 19 0E BD F0 C6 7C 9E 4D 26 C7 41 A5
   Counter block (2): 00 1C C5 B7 51 A5 1D 70 A1 C1 11 48 00 00 00 02
   Key stream    (2): A4 16 CD 95 71 7C EB 10 EC 95 DA AE 9F CB 19 00
   Counter block (3): 00 1C C5 B7 51 A5 1D 70 A1 C1 11 48 00 00 00 03
   Key stream    (3): 3E E1 C4 9B C6 B9 CA 21 3F 6E E2 71 D0 A9 33 39
   Ciphertext       : EB 6C 52 82 1D 0B BB F7 CE 75 94 46 2A CA 4F AA
                    : B4 07 DF 86 65 69 FD 07 F4 8C C0 B5 83 D6 07 1F
                    : 1E C0 E6 B8
*/
static const uint8_t K9_CTR[] = {
        0xFF, 0x7A, 0x61, 0x7C, 0xE6, 0x91, 0x48, 0xE4,
        0xF1, 0x72, 0x6E, 0x2F, 0x43, 0x58, 0x1D, 0xE2,
        0xAA, 0x62, 0xD9, 0xF8, 0x05, 0x53, 0x2E, 0xDF,
        0xF1, 0xEE, 0xD6, 0x87, 0xFB, 0x54, 0x15, 0x3D,
};
static const uint8_t IV9_CTR[] = {
        0x00, 0x1C, 0xC5, 0xB7,	/* nonce */
        0x51, 0xA5, 0x1D, 0x70, 0xA1, 0xC1, 0x11, 0x48,
};
static const uint8_t P9_CTR[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
        0x20, 0x21, 0x22, 0x23,
};
static const uint8_t C9_CTR[] = {
        0xEB, 0x6C, 0x52, 0x82, 0x1D, 0x0B, 0xBB, 0xF7,
        0xCE, 0x75, 0x94, 0x46, 0x2A, 0xCA, 0x4F, 0xAA,
        0xB4, 0x07, 0xDF, 0x86, 0x65, 0x69, 0xFD, 0x07,
        0xF4, 0x8C, 0xC0, 0xB5, 0x83, 0xD6, 0x07, 0x1F,
        0x1E, 0xC0, 0xE6, 0xB8,
};
static const uint8_t T9_CTR[] = { 0 };
static const uint8_t A9_CTR[] = { 0 };
#define A9_CTR_len 0

/*
   Test Vector #10: Encrypting 128 octets using AES-CTR with 128-bit key
   AES Key          : AE 68 52 F8 12 10 67 CC 4B F7 A5 76 55 77 F3 9E
   AES-CTR IV       : 00 00 00 00 00 00 00 00
   Nonce            : 00 00 00 30
   Plaintext String : 'Full 8 blocks'
*/
static const uint8_t K10_CTR[] = {
        0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
        0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E,
};
static const uint8_t IV10_CTR[] = {
        0x00, 0x00, 0x00, 0x30,	/* nonce */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};
static const uint8_t P10_CTR[] = {
        0xA8, 0x63, 0x44, 0xF8, 0x36, 0x59, 0x2F, 0xF2,
        0xDA, 0xDD, 0x17, 0xCE, 0xFE, 0x2A, 0xF2, 0xA2,
        0x35, 0x87, 0x34, 0x0F, 0x35, 0xFC, 0xD8, 0xF2,
        0x57, 0xA1, 0xCB, 0x19, 0x0C, 0x33, 0x14, 0xE1,
        0x23, 0xEB, 0xC0, 0x88, 0x82, 0x05, 0x5F, 0x01,
        0x5D, 0xFC, 0x53, 0x08, 0xDB, 0x34, 0x8E, 0x94,
        0xE4, 0xA8, 0x26, 0x7F, 0xBC, 0xB7, 0x8B, 0xE1,
        0x58, 0x2F, 0x2C, 0x91, 0xCD, 0x5B, 0x4A, 0xAA,
        0x7A, 0xBA, 0x5F, 0xD2, 0x9B, 0xF8, 0x7D, 0xEA,
        0x76, 0xB6, 0x64, 0xB3, 0x29, 0xD3, 0x02, 0x19,
        0xA0, 0xDC, 0xE9, 0xB8, 0x90, 0x51, 0xA8, 0xDE,
        0x2E, 0xA1, 0xB7, 0x7E, 0x51, 0x0D, 0x34, 0xB3,
        0xED, 0xE7, 0x5E, 0xB8, 0x8A, 0xE9, 0xFE, 0x89,
        0xF8, 0x0B, 0x85, 0x09, 0x76, 0x08, 0x78, 0x0D,
        0x27, 0x59, 0x8E, 0x14, 0x43, 0x46, 0xA0, 0x91,
        0xEE, 0xAA, 0xFF, 0x74, 0x8D, 0xBC, 0x98, 0xB9
};

static const uint8_t C10_CTR[] = {
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A
};

static const uint8_t T10_CTR[] = { 0 };
static const uint8_t A10_CTR[] = { 0 };
#define A10_CTR_len 0

/*
   Test Vector #11: Encrypting 136 octets using AES-CTR with 128-bit key
   AES Key          : AE 68 52 F8 12 10 67 CC 4B F7 A5 76 55 77 F3 9E
   AES-CTR IV       : 00 00 00 00 00 00 00 00
   Nonce            : 00 00 00 30
   Plaintext String : 'Full 8 blocks + extra partial block'
*/
static const uint8_t K11_CTR[] = {
        0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
        0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E,
};
static const uint8_t IV11_CTR[] = {
        0x00, 0x00, 0x00, 0x30,	/* nonce */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};
static const uint8_t P11_CTR[] = {
        0xA8, 0x63, 0x44, 0xF8, 0x36, 0x59, 0x2F, 0xF2,
        0xDA, 0xDD, 0x17, 0xCE, 0xFE, 0x2A, 0xF2, 0xA2,
        0x35, 0x87, 0x34, 0x0F, 0x35, 0xFC, 0xD8, 0xF2,
        0x57, 0xA1, 0xCB, 0x19, 0x0C, 0x33, 0x14, 0xE1,
        0x23, 0xEB, 0xC0, 0x88, 0x82, 0x05, 0x5F, 0x01,
        0x5D, 0xFC, 0x53, 0x08, 0xDB, 0x34, 0x8E, 0x94,
        0xE4, 0xA8, 0x26, 0x7F, 0xBC, 0xB7, 0x8B, 0xE1,
        0x58, 0x2F, 0x2C, 0x91, 0xCD, 0x5B, 0x4A, 0xAA,
        0x7A, 0xBA, 0x5F, 0xD2, 0x9B, 0xF8, 0x7D, 0xEA,
        0x76, 0xB6, 0x64, 0xB3, 0x29, 0xD3, 0x02, 0x19,
        0xA0, 0xDC, 0xE9, 0xB8, 0x90, 0x51, 0xA8, 0xDE,
        0x2E, 0xA1, 0xB7, 0x7E, 0x51, 0x0D, 0x34, 0xB3,
        0xED, 0xE7, 0x5E, 0xB8, 0x8A, 0xE9, 0xFE, 0x89,
        0xF8, 0x0B, 0x85, 0x09, 0x76, 0x08, 0x78, 0x0D,
        0x27, 0x59, 0x8E, 0x14, 0x43, 0x46, 0xA0, 0x91,
        0xEE, 0xAA, 0xFF, 0x74, 0x8D, 0xBC, 0x98, 0xB9,
        0x12, 0xAD, 0x82, 0xDF, 0x2F, 0xF8, 0x9C, 0xE0
};

static const uint8_t C11_CTR[] = {
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A,
        0x7A, 0x13, 0xB4, 0x40, 0x69, 0x9B, 0x58, 0x16
};
static const uint8_t T11_CTR[] = { 0 };
static const uint8_t A11_CTR[] = { 0 };
#define A11_CTR_len 0

/*
   Test Vector #12: Encrypting 256 octets using AES-CTR with 128-bit key
   AES Key          : AE 68 52 F8 12 10 67 CC 4B F7 A5 76 55 77 F3 9E
   AES-CTR IV       : 00 00 00 00 00 00 00 00
   Nonce            : 00 00 00 30
   Plaintext String : 'Full 2x8 blocks'
*/
static const uint8_t K12_CTR[] = {
        0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
        0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E,
};
static const uint8_t IV12_CTR[] = {
        0x00, 0x00, 0x00, 0x30,	/* nonce */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};
static const uint8_t P12_CTR[] = {
        0xA8, 0x63, 0x44, 0xF8, 0x36, 0x59, 0x2F, 0xF2,
        0xDA, 0xDD, 0x17, 0xCE, 0xFE, 0x2A, 0xF2, 0xA2,
        0x35, 0x87, 0x34, 0x0F, 0x35, 0xFC, 0xD8, 0xF2,
        0x57, 0xA1, 0xCB, 0x19, 0x0C, 0x33, 0x14, 0xE1,
        0x23, 0xEB, 0xC0, 0x88, 0x82, 0x05, 0x5F, 0x01,
        0x5D, 0xFC, 0x53, 0x08, 0xDB, 0x34, 0x8E, 0x94,
        0xE4, 0xA8, 0x26, 0x7F, 0xBC, 0xB7, 0x8B, 0xE1,
        0x58, 0x2F, 0x2C, 0x91, 0xCD, 0x5B, 0x4A, 0xAA,
        0x7A, 0xBA, 0x5F, 0xD2, 0x9B, 0xF8, 0x7D, 0xEA,
        0x76, 0xB6, 0x64, 0xB3, 0x29, 0xD3, 0x02, 0x19,
        0xA0, 0xDC, 0xE9, 0xB8, 0x90, 0x51, 0xA8, 0xDE,
        0x2E, 0xA1, 0xB7, 0x7E, 0x51, 0x0D, 0x34, 0xB3,
        0xED, 0xE7, 0x5E, 0xB8, 0x8A, 0xE9, 0xFE, 0x89,
        0xF8, 0x0B, 0x85, 0x09, 0x76, 0x08, 0x78, 0x0D,
        0x27, 0x59, 0x8E, 0x14, 0x43, 0x46, 0xA0, 0x91,
        0xEE, 0xAA, 0xFF, 0x74, 0x8D, 0xBC, 0x98, 0xB9,
        0x77, 0xBD, 0x41, 0x4F, 0xAB, 0xF8, 0x78, 0x1F,
        0xED, 0x2B, 0x14, 0x89, 0xB5, 0x7B, 0x61, 0x5E,
        0x88, 0x35, 0x46, 0x0F, 0x83, 0x5B, 0xC6, 0xE6,
        0x61, 0x1D, 0xD8, 0x5E, 0xD3, 0xC3, 0xC6, 0xE8,
        0xFB, 0x8E, 0x59, 0xDB, 0x31, 0x17, 0xF8, 0xCD,
        0xC1, 0xD4, 0x2D, 0xEF, 0xD8, 0x25, 0x9E, 0x88,
        0x10, 0x58, 0xF2, 0xA6, 0x84, 0x4F, 0xA1, 0x32,
        0x5F, 0x0E, 0xA2, 0x14, 0xF7, 0x03, 0x85, 0x06,
        0x94, 0x4F, 0x83, 0x87, 0x04, 0x97, 0x5A, 0x8D,
        0x9A, 0x73, 0x36, 0x2A, 0xE0, 0x6D, 0xA9, 0x1F,
        0xBC, 0x2F, 0xD2, 0x9E, 0xD1, 0x7D, 0x2C, 0x89,
        0x1F, 0xE1, 0xA0, 0x8F, 0x5D, 0x3E, 0xAB, 0x9E,
        0x79, 0x1A, 0x76, 0xC3, 0x0A, 0xC8, 0xCF, 0xCB,
        0x35, 0x63, 0xD9, 0x46, 0x87, 0xAF, 0x74, 0x24,
        0x47, 0xBA, 0x60, 0xAB, 0x33, 0x5D, 0xA8, 0xDE,
        0xFE, 0x1B, 0xC5, 0x3F, 0xAC, 0xD9, 0xAD, 0x94
};

static const uint8_t C12_CTR[] = {
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A,
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A,
};
static const uint8_t T12_CTR[] = { 0 };
static const uint8_t A12_CTR[] = { 0 };
#define A12_CTR_len 0

/*
   Test Vector #13: Encrypting 300 octets using AES-CTR with 128-bit key
   AES Key          : AE 68 52 F8 12 10 67 CC 4B F7 A5 76 55 77 F3 9E
   AES-CTR IV       : 00 00 00 00 00 00 00 00
   Nonce            : 00 00 00 30
   Plaintext String : 'Full 2x8 blocks + 44B (2 full blocks and partial block)'
*/
static const uint8_t K13_CTR[] = {
        0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
        0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E,
};
static const uint8_t IV13_CTR[] = {
        0x00, 0x00, 0x00, 0x30,	/* nonce */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};
static const uint8_t P13_CTR[] = {
        0xA8, 0x63, 0x44, 0xF8, 0x36, 0x59, 0x2F, 0xF2,
        0xDA, 0xDD, 0x17, 0xCE, 0xFE, 0x2A, 0xF2, 0xA2,
        0x35, 0x87, 0x34, 0x0F, 0x35, 0xFC, 0xD8, 0xF2,
        0x57, 0xA1, 0xCB, 0x19, 0x0C, 0x33, 0x14, 0xE1,
        0x23, 0xEB, 0xC0, 0x88, 0x82, 0x05, 0x5F, 0x01,
        0x5D, 0xFC, 0x53, 0x08, 0xDB, 0x34, 0x8E, 0x94,
        0xE4, 0xA8, 0x26, 0x7F, 0xBC, 0xB7, 0x8B, 0xE1,
        0x58, 0x2F, 0x2C, 0x91, 0xCD, 0x5B, 0x4A, 0xAA,
        0x7A, 0xBA, 0x5F, 0xD2, 0x9B, 0xF8, 0x7D, 0xEA,
        0x76, 0xB6, 0x64, 0xB3, 0x29, 0xD3, 0x02, 0x19,
        0xA0, 0xDC, 0xE9, 0xB8, 0x90, 0x51, 0xA8, 0xDE,
        0x2E, 0xA1, 0xB7, 0x7E, 0x51, 0x0D, 0x34, 0xB3,
        0xED, 0xE7, 0x5E, 0xB8, 0x8A, 0xE9, 0xFE, 0x89,
        0xF8, 0x0B, 0x85, 0x09, 0x76, 0x08, 0x78, 0x0D,
        0x27, 0x59, 0x8E, 0x14, 0x43, 0x46, 0xA0, 0x91,
        0xEE, 0xAA, 0xFF, 0x74, 0x8D, 0xBC, 0x98, 0xB9,
        0x77, 0xBD, 0x41, 0x4F, 0xAB, 0xF8, 0x78, 0x1F,
        0xED, 0x2B, 0x14, 0x89, 0xB5, 0x7B, 0x61, 0x5E,
        0x88, 0x35, 0x46, 0x0F, 0x83, 0x5B, 0xC6, 0xE6,
        0x61, 0x1D, 0xD8, 0x5E, 0xD3, 0xC3, 0xC6, 0xE8,
        0xFB, 0x8E, 0x59, 0xDB, 0x31, 0x17, 0xF8, 0xCD,
        0xC1, 0xD4, 0x2D, 0xEF, 0xD8, 0x25, 0x9E, 0x88,
        0x10, 0x58, 0xF2, 0xA6, 0x84, 0x4F, 0xA1, 0x32,
        0x5F, 0x0E, 0xA2, 0x14, 0xF7, 0x03, 0x85, 0x06,
        0x94, 0x4F, 0x83, 0x87, 0x04, 0x97, 0x5A, 0x8D,
        0x9A, 0x73, 0x36, 0x2A, 0xE0, 0x6D, 0xA9, 0x1F,
        0xBC, 0x2F, 0xD2, 0x9E, 0xD1, 0x7D, 0x2C, 0x89,
        0x1F, 0xE1, 0xA0, 0x8F, 0x5D, 0x3E, 0xAB, 0x9E,
        0x79, 0x1A, 0x76, 0xC3, 0x0A, 0xC8, 0xCF, 0xCB,
        0x35, 0x63, 0xD9, 0x46, 0x87, 0xAF, 0x74, 0x24,
        0x47, 0xBA, 0x60, 0xAB, 0x33, 0x5D, 0xA8, 0xDE,
        0xFE, 0x1B, 0xC5, 0x3F, 0xAC, 0xD9, 0xAD, 0x94,
        0x66, 0xb8, 0x3f, 0x3a, 0x21, 0x9f, 0xd0, 0x43,
        0x46, 0xdd, 0x65, 0x8b, 0x44, 0x99, 0x66, 0x91,
        0x64, 0xe2, 0x69, 0x6f, 0xbb, 0x85, 0x8c, 0xcc,
        0x7f, 0xea, 0x96, 0xd1, 0x5e, 0xb4, 0x7c, 0xd0,
        0xab, 0x02, 0x8d, 0xa3, 0x59, 0x3b, 0x8c, 0xd5,
        0xd0, 0xe7, 0xb4, 0xc4
};

static const uint8_t C13_CTR[] = {
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A,
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A,
        0xED, 0x52, 0x55, 0xB9, 0x76, 0x6C, 0x5E, 0x6E,
        0x76, 0x97, 0x00, 0xC7, 0xEB, 0xFE, 0xEC, 0x10,
        0x94, 0x2C, 0xA9, 0xAF, 0x9B, 0x09, 0x19, 0xB3,
        0x17, 0x29, 0x96, 0xBA, 0x8E, 0xAC, 0x3D, 0x0A,
        0x9B, 0x70, 0x54, 0x0F, 0x1E, 0xD4, 0xE8, 0x13,
        0xE6, 0x8F, 0xAD, 0xFD

};
static const uint8_t T13_CTR[] = { 0 };
static const uint8_t A13_CTR[] = { 0 };
#define A13_CTR_len 0

/*
   Test Vector #14: Encrypting 384 octets using AES-CTR with 128-bit key
   AES Key          : AE 68 52 F8 12 10 67 CC 4B F7 A5 76 55 77 F3 9E
   AES-CTR IV       : 00 00 00 00 00 00 00 00
   Nonce            : 00 00 00 30
   Plaintext String : 'Full 3x8 blocks'
*/
static const uint8_t K14_CTR[] = {
        0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
        0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E,
};
static const uint8_t IV14_CTR[] = {
        0x00, 0x00, 0x00, 0x30,	/* nonce */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};
static const uint8_t P14_CTR[] = {
        0xA8, 0x63, 0x44, 0xF8, 0x36, 0x59, 0x2F, 0xF2,
        0xDA, 0xDD, 0x17, 0xCE, 0xFE, 0x2A, 0xF2, 0xA2,
        0x35, 0x87, 0x34, 0x0F, 0x35, 0xFC, 0xD8, 0xF2,
        0x57, 0xA1, 0xCB, 0x19, 0x0C, 0x33, 0x14, 0xE1,
        0x23, 0xEB, 0xC0, 0x88, 0x82, 0x05, 0x5F, 0x01,
        0x5D, 0xFC, 0x53, 0x08, 0xDB, 0x34, 0x8E, 0x94,
        0xE4, 0xA8, 0x26, 0x7F, 0xBC, 0xB7, 0x8B, 0xE1,
        0x58, 0x2F, 0x2C, 0x91, 0xCD, 0x5B, 0x4A, 0xAA,
        0x7A, 0xBA, 0x5F, 0xD2, 0x9B, 0xF8, 0x7D, 0xEA,
        0x76, 0xB6, 0x64, 0xB3, 0x29, 0xD3, 0x02, 0x19,
        0xA0, 0xDC, 0xE9, 0xB8, 0x90, 0x51, 0xA8, 0xDE,
        0x2E, 0xA1, 0xB7, 0x7E, 0x51, 0x0D, 0x34, 0xB3,
        0xED, 0xE7, 0x5E, 0xB8, 0x8A, 0xE9, 0xFE, 0x89,
        0xF8, 0x0B, 0x85, 0x09, 0x76, 0x08, 0x78, 0x0D,
        0x27, 0x59, 0x8E, 0x14, 0x43, 0x46, 0xA0, 0x91,
        0xEE, 0xAA, 0xFF, 0x74, 0x8D, 0xBC, 0x98, 0xB9,
        0x77, 0xBD, 0x41, 0x4F, 0xAB, 0xF8, 0x78, 0x1F,
        0xED, 0x2B, 0x14, 0x89, 0xB5, 0x7B, 0x61, 0x5E,
        0x88, 0x35, 0x46, 0x0F, 0x83, 0x5B, 0xC6, 0xE6,
        0x61, 0x1D, 0xD8, 0x5E, 0xD3, 0xC3, 0xC6, 0xE8,
        0xFB, 0x8E, 0x59, 0xDB, 0x31, 0x17, 0xF8, 0xCD,
        0xC1, 0xD4, 0x2D, 0xEF, 0xD8, 0x25, 0x9E, 0x88,
        0x10, 0x58, 0xF2, 0xA6, 0x84, 0x4F, 0xA1, 0x32,
        0x5F, 0x0E, 0xA2, 0x14, 0xF7, 0x03, 0x85, 0x06,
        0x94, 0x4F, 0x83, 0x87, 0x04, 0x97, 0x5A, 0x8D,
        0x9A, 0x73, 0x36, 0x2A, 0xE0, 0x6D, 0xA9, 0x1F,
        0xBC, 0x2F, 0xD2, 0x9E, 0xD1, 0x7D, 0x2C, 0x89,
        0x1F, 0xE1, 0xA0, 0x8F, 0x5D, 0x3E, 0xAB, 0x9E,
        0x79, 0x1A, 0x76, 0xC3, 0x0A, 0xC8, 0xCF, 0xCB,
        0x35, 0x63, 0xD9, 0x46, 0x87, 0xAF, 0x74, 0x24,
        0x47, 0xBA, 0x60, 0xAB, 0x33, 0x5D, 0xA8, 0xDE,
        0xFE, 0x1B, 0xC5, 0x3F, 0xAC, 0xD9, 0xAD, 0x94,
        0x66, 0xb8, 0x3f, 0x3a, 0x21, 0x9f, 0xd0, 0x43,
        0x46, 0xdd, 0x65, 0x8b, 0x44, 0x99, 0x66, 0x91,
        0x64, 0xe2, 0x69, 0x6f, 0xbb, 0x85, 0x8c, 0xcc,
        0x7f, 0xea, 0x96, 0xd1, 0x5e, 0xb4, 0x7c, 0xd0,
        0xab, 0x02, 0x8d, 0xa3, 0x59, 0x3b, 0x8c, 0xd5,
        0xd0, 0xe7, 0xb4, 0xc4, 0x90, 0x41, 0x9f, 0x78,
        0x4e, 0x82, 0x9e, 0xe4, 0x1b, 0x97, 0xa9, 0xa4,
        0x7b, 0x48, 0xad, 0x56, 0xc0, 0xe4, 0x86, 0x52,
        0xfc, 0xad, 0x93, 0x0b, 0x7d, 0x38, 0xce, 0x73,
        0x64, 0xbd, 0xf7, 0x00, 0x7b, 0xe6, 0x46, 0x03,
        0x2f, 0x4b, 0x75, 0x9f, 0x3a, 0x2d, 0x32, 0x42,
        0xfe, 0x80, 0x74, 0x89, 0x27, 0x34, 0xce, 0x5e,
        0xbf, 0xbe, 0x07, 0x50, 0x91, 0x08, 0x27, 0x2b,
        0x32, 0x77, 0xa7, 0xff, 0x83, 0xb1, 0xab, 0xc8,
        0x98, 0xbe, 0xac, 0x33, 0x7c, 0x47, 0x19, 0x33,
        0x6f, 0x4d, 0xbe, 0x3e, 0xdc, 0xe0, 0x87, 0xfb
};

static const uint8_t C14_CTR[] = {
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A,
        0x1F, 0x03, 0x77, 0xD0, 0xED, 0x9B, 0xBC, 0xE9,
        0x9B, 0xD3, 0x01, 0x06, 0xF8, 0x54, 0x90, 0x7D,
        0x67, 0x6F, 0x19, 0xD7, 0x0B, 0xF3, 0x92, 0x8D,
        0x60, 0xC0, 0x18, 0x5A, 0x24, 0xC1, 0xD7, 0x60,
        0x82, 0x9C, 0x22, 0x37, 0x45, 0xE3, 0x9D, 0xA6,
        0x76, 0x37, 0xE1, 0x7A, 0x13, 0xB4, 0x40, 0x63,
        0xF4, 0xD8, 0xDE, 0x41, 0x64, 0xFC, 0xE2, 0x42,
        0x2E, 0x3F, 0xEA, 0xE1, 0x28, 0x06, 0xA5, 0xAC,
        0x6A, 0xC1, 0x58, 0x0C, 0x84, 0xFF, 0x9B, 0x6A,
        0xE5, 0xBE, 0x4E, 0x8C, 0x4C, 0xE9, 0x97, 0xD5,
        0x24, 0x30, 0x1B, 0x19, 0xDF, 0x87, 0x56, 0x85,
        0x31, 0x56, 0x5A, 0xDE, 0xE0, 0x6E, 0xC0, 0x1C,
        0xCB, 0x51, 0x5B, 0x6E, 0xAC, 0xF5, 0xB0, 0x60,
        0x60, 0x2F, 0x62, 0x0A, 0xEA, 0x62, 0x51, 0x2E,
        0x5B, 0x1B, 0x99, 0x51, 0x3B, 0xAC, 0xE9, 0xC5,
        0x59, 0x7D, 0x0E, 0xB6, 0x51, 0x6C, 0x16, 0x7A,
        0xED, 0x52, 0x55, 0xB9, 0x76, 0x6C, 0x5E, 0x6E,
        0x76, 0x97, 0x00, 0xC7, 0xEB, 0xFE, 0xEC, 0x10,
        0x94, 0x2C, 0xA9, 0xAF, 0x9B, 0x09, 0x19, 0xB3,
        0x17, 0x29, 0x96, 0xBA, 0x8E, 0xAC, 0x3D, 0x0A,
        0x9B, 0x70, 0x54, 0x0F, 0x1E, 0xD4, 0xE8, 0x13,
        0xE6, 0x8F, 0xAD, 0xFD, 0xFD, 0x13, 0xCF, 0xD5,
        0x94, 0x06, 0xA0, 0x24, 0x79, 0xC0, 0xF8, 0x05,
        0x3D, 0x19, 0xEB, 0x96, 0xDA, 0x31, 0xAE, 0xF5,
        0x4D, 0x82, 0x2C, 0x23, 0x03, 0x9A, 0x43, 0x85,
        0x94, 0x36, 0x30, 0xE8, 0x0A, 0x9B, 0x1F, 0x05,
        0x6E, 0x4B, 0xA5, 0x98, 0x78, 0xBE, 0x73, 0x0D,
        0x8C, 0x60, 0x55, 0x88, 0xD6, 0xA3, 0x80, 0x13,
        0x19, 0xDB, 0xF8, 0xCD, 0xA7, 0xDC, 0x28, 0x4C,
        0x09, 0xAF, 0xFE, 0x88, 0x77, 0xE1, 0x6E, 0x12,
        0x57, 0x5A, 0xA8, 0xC6, 0x38, 0xCF, 0xF5, 0x0D,
        0x42, 0x2C, 0x67, 0xB3, 0x22, 0x6F, 0x3D, 0x7D
};
static const uint8_t T14_CTR[] = { 0 };
static const uint8_t A14_CTR[] = { 0 };
#define A14_CTR_len 0

static const uint8_t K15_CTR[] = {
        0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
        0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
};
static const uint8_t IV15_CTR[] = {
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
        0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
};
static const uint8_t P15_CTR[] = {
        0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
        0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A,
        0xAE, 0x2D, 0x8A, 0x57, 0x1E, 0x03, 0xAC, 0x9C,
        0x9E, 0xB7, 0x6F, 0xAC, 0x45, 0xAF, 0x8E, 0x51,
        0x30, 0xC8, 0x1C, 0x46, 0xA3, 0x5C, 0xE4, 0x11,
        0xE5, 0xFB, 0xC1, 0x19, 0x1A, 0x0A, 0x52, 0xEF,
        0xF6, 0x9F, 0x24, 0x45, 0xDF, 0x4F, 0x9B, 0x17,
        0xAD, 0x2B, 0x41, 0x7B, 0xE6, 0x6C, 0x37, 0x10
};

static const uint8_t C15_CTR[] = {
        0x87, 0x4D, 0x61, 0x91, 0xB6, 0x20, 0xE3, 0x26,
        0x1B, 0xEF, 0x68, 0x64, 0x99, 0x0D, 0xB6, 0xCE,
        0x98, 0x06, 0xF6, 0x6B, 0x79, 0x70, 0xFD, 0xFF,
        0x86, 0x17, 0x18, 0x7B, 0xB9, 0xFF, 0xFD, 0xFF,
        0x5A, 0xE4, 0xDF, 0x3E, 0xDB, 0xD5, 0xD3, 0x5E,
        0x5B, 0x4F, 0x09, 0x02, 0x0D, 0xB0, 0x3E, 0xAB,
        0x1E, 0x03, 0x1D, 0xDA, 0x2F, 0xBE, 0x03, 0xD1,
        0x79, 0x21, 0x70, 0xA0, 0xF3, 0x00, 0x9C, 0xEE
};

static const uint8_t T15_CTR[] = { 0 };
static const uint8_t A15_CTR[] = { 0 };
#define A15_CTR_len 0

#define bit_vector(N)                                                 \
        {K##N, (KBITS(K##N)), IV##N, sizeof(IV##N), A##N, A##N##_len, \
                        P##N, P##N##_len, C##N, T##N, sizeof(T##N)}

static const uint8_t K1_CTR_BIT[] = {
        0xd3, 0xc5, 0xd5, 0x92, 0x32, 0x7f, 0xb1, 0x1c,
        0x40, 0x35, 0xc6, 0x68, 0x0a, 0xf8, 0xc6, 0xd1

};
static const uint8_t IV1_CTR_BIT[] = {
        0x39, 0x8a, 0x59, 0xb4, 0xac, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Last 3 bits (not to be ciphered) set to 1, since
 * the output buffer is set to all 0xff */
static const uint8_t P1_CTR_BIT[] = {
        0x98, 0x1b, 0xa6, 0x82, 0x4c, 0x1b, 0xfb, 0x1a,
        0xb4, 0x85, 0x47, 0x20, 0x29, 0xb7, 0x1d, 0x80,
        0x8c, 0xe3, 0x3e, 0x2c, 0xc3, 0xc0, 0xb5, 0xfc,
        0x1f, 0x3d, 0xe8, 0xa6, 0xdc, 0x66, 0xb1, 0xf7
};

static const uint8_t C1_CTR_BIT[] = {
        0xe9, 0xfe, 0xd8, 0xa6, 0x3d, 0x15, 0x53, 0x04,
        0xd7, 0x1d, 0xf2, 0x0b, 0xf3, 0xe8, 0x22, 0x14,
        0xb2, 0x0e, 0xd7, 0xda, 0xd2, 0xf2, 0x33, 0xdc,
        0x3c, 0x22, 0xd7, 0xbd, 0xee, 0xed, 0x8e, 0x7f
};

static const uint8_t T1_CTR_BIT[] = { 0 };
static const uint8_t A1_CTR_BIT[] = { 0 };
#define A1_CTR_BIT_len 0
#define P1_CTR_BIT_len 253

static const uint8_t K2_CTR_BIT[] = {
        0x2b, 0xd6, 0x45, 0x9f, 0x82, 0xc4, 0x40, 0xe0,
        0x95, 0x2c, 0x49, 0x10, 0x48, 0x05, 0xff, 0x48
};

static const uint8_t IV2_CTR_BIT[] = {
        0xc6, 0x75, 0xa6, 0x4b, 0x64, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Last 2 bits (not to be ciphered) set to 1, since
 * the output buffer is set to all 0xff */
static const uint8_t P2_CTR_BIT[] = {
        0x7e, 0xc6, 0x12, 0x72, 0x74, 0x3b, 0xf1, 0x61,
        0x47, 0x26, 0x44, 0x6a, 0x6c, 0x38, 0xce, 0xd1,
        0x66, 0xf6, 0xca, 0x76, 0xeb, 0x54, 0x30, 0x04,
        0x42, 0x86, 0x34, 0x6c, 0xef, 0x13, 0x0f, 0x92,
        0x92, 0x2b, 0x03, 0x45, 0x0d, 0x3a, 0x99, 0x75,
        0xe5, 0xbd, 0x2e, 0xa0, 0xeb, 0x55, 0xad, 0x8e,
        0x1b, 0x19, 0x9e, 0x3e, 0xc4, 0x31, 0x60, 0x20,
        0xe9, 0xa1, 0xb2, 0x85, 0xe7, 0x62, 0x79, 0x53,
        0x59, 0xb7, 0xbd, 0xfd, 0x39, 0xbe, 0xf4, 0xb2,
        0x48, 0x45, 0x83, 0xd5, 0xaf, 0xe0, 0x82, 0xae,
        0xe6, 0x38, 0xbf, 0x5f, 0xd5, 0xa6, 0x06, 0x19,
        0x39, 0x01, 0xa0, 0x8f, 0x4a, 0xb4, 0x1a, 0xab,
        0x9b, 0x13, 0x48, 0x83
};

static const uint8_t C2_CTR_BIT[] = {
        0x59, 0x61, 0x60, 0x53, 0x53, 0xc6, 0x4b, 0xdc,
        0xa1, 0x5b, 0x19, 0x5e, 0x28, 0x85, 0x53, 0xa9,
        0x10, 0x63, 0x25, 0x06, 0xd6, 0x20, 0x0a, 0xa7,
        0x90, 0xc4, 0xc8, 0x06, 0xc9, 0x99, 0x04, 0xcf,
        0x24, 0x45, 0xcc, 0x50, 0xbb, 0x1c, 0xf1, 0x68,
        0xa4, 0x96, 0x73, 0x73, 0x4e, 0x08, 0x1b, 0x57,
        0xe3, 0x24, 0xce, 0x52, 0x59, 0xc0, 0xe7, 0x8d,
        0x4c, 0xd9, 0x7b, 0x87, 0x09, 0x76, 0x50, 0x3c,
        0x09, 0x43, 0xf2, 0xcb, 0x5a, 0xe8, 0xf0, 0x52,
        0xc7, 0xb7, 0xd3, 0x92, 0x23, 0x95, 0x87, 0xb8,
        0x95, 0x60, 0x86, 0xbc, 0xab, 0x18, 0x83, 0x60,
        0x42, 0xe2, 0xe6, 0xce, 0x42, 0x43, 0x2a, 0x17,
        0x10, 0x5c, 0x53, 0xd3
};

static const uint8_t T2_CTR_BIT[] = { 0 };
static const uint8_t A2_CTR_BIT[] = { 0 };
#define A2_CTR_BIT_len 0
#define P2_CTR_BIT_len 798

static const uint8_t K3_CTR_BIT[] = {
        0x0a, 0x8b, 0x6b, 0xd8, 0xd9, 0xb0, 0x8b, 0x08,
        0xd6, 0x4e, 0x32, 0xd1, 0x81, 0x77, 0x77, 0xfb
};

static const uint8_t IV3_CTR_BIT[] = {
        0x54, 0x4d, 0x49, 0xcd, 0x20, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Last 2 bits (not to be ciphered) set to 1, since
 * the output buffer is set to all 0xff */
static const uint8_t P3_CTR_BIT[] = {
        0xfd, 0x40, 0xa4, 0x1d, 0x37, 0x0a, 0x1f, 0x65,
        0x74, 0x50, 0x95, 0x68, 0x7d, 0x47, 0xba, 0x1d,
        0x36, 0xd2, 0x34, 0x9e, 0x23, 0xf6, 0x44, 0x39,
        0x2c, 0x8e, 0xa9, 0xc4, 0x9d, 0x40, 0xc1, 0x32,
        0x71, 0xaf, 0xf2, 0x64, 0xd0, 0xf2, 0x4b
};

static const uint8_t C3_CTR_BIT[] = {
        0x75, 0x75, 0x0d, 0x37, 0xb4, 0xbb, 0xa2, 0xa4,
        0xde, 0xdb, 0x34, 0x23, 0x5b, 0xd6, 0x8c, 0x66,
        0x45, 0xac, 0xda, 0xac, 0xa4, 0x81, 0x38, 0xa3,
        0xb0, 0xc4, 0x71, 0xe2, 0xa7, 0x04, 0x1a, 0x57,
        0x64, 0x23, 0xd2, 0x92, 0x72, 0x87, 0xf3
};

static const uint8_t T3_CTR_BIT[] = { 0 };
static const uint8_t A3_CTR_BIT[] = { 0 };
#define A3_CTR_BIT_len 0
#define P3_CTR_BIT_len 310

static const uint8_t K4_CTR_BIT[] = {
        0xaa, 0x1f, 0x95, 0xae, 0xa5, 0x33, 0xbc, 0xb3,
        0x2e, 0xb6, 0x3b, 0xf5, 0x2d, 0x8f, 0x83, 0x1a
};

static const uint8_t IV4_CTR_BIT[] = {
        0x72, 0xd8, 0xc6, 0x71, 0x84, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Last 2 bits (not to be ciphered) set to 1, since
 * the output buffer is set to all 0xff */
static const uint8_t P4_CTR_BIT[] = {
        0xfb, 0x1b, 0x96, 0xc5, 0xc8, 0xba, 0xdf, 0xb2,
        0xe8, 0xe8, 0xed, 0xfd, 0xe7, 0x8e, 0x57, 0xf2,
        0xad, 0x81, 0xe7, 0x41, 0x03, 0xfc, 0x43, 0x0a,
        0x53, 0x4d, 0xcc, 0x37, 0xaf, 0xce, 0xc7, 0x0e,
        0x15, 0x17, 0xbb, 0x06, 0xf2, 0x72, 0x19, 0xda,
        0xe4, 0x90, 0x22, 0xdd, 0xc4, 0x7a, 0x06, 0x8d,
        0xe4, 0xc9, 0x49, 0x6a, 0x95, 0x1a, 0x6b, 0x09,
        0xed, 0xbd, 0xc8, 0x64, 0xc7, 0xad, 0xbd, 0x74,
        0x0a, 0xc5, 0x0c, 0x02, 0x2f, 0x30, 0x82, 0xba,
        0xfd, 0x22, 0xd7, 0x81, 0x97, 0xc5, 0xd5, 0x08,
        0xb9, 0x77, 0xbc, 0xa1, 0x3f, 0x32, 0xe6, 0x52,
        0xe7, 0x4b, 0xa7, 0x28, 0x57, 0x60, 0x77, 0xce,
        0x62, 0x8c, 0x53, 0x5e, 0x87, 0xdc, 0x60, 0x77,
        0xba, 0x07, 0xd2, 0x90, 0x68, 0x59, 0x0c, 0x8c,
        0xb5, 0xf1, 0x08, 0x8e, 0x08, 0x2c, 0xfa, 0x0e,
        0xc9, 0x61, 0x30, 0x2d, 0x69, 0xcf, 0x3d, 0x47
};

static const uint8_t C4_CTR_BIT[] = {
        0xdf, 0xb4, 0x40, 0xac, 0xb3, 0x77, 0x35, 0x49,
        0xef, 0xc0, 0x46, 0x28, 0xae, 0xb8, 0xd8, 0x15,
        0x62, 0x75, 0x23, 0x0b, 0xdc, 0x69, 0x0d, 0x94,
        0xb0, 0x0d, 0x8d, 0x95, 0xf2, 0x8c, 0x4b, 0x56,
        0x30, 0x7f, 0x60, 0xf4, 0xca, 0x55, 0xeb, 0xa6,
        0x61, 0xeb, 0xba, 0x72, 0xac, 0x80, 0x8f, 0xa8,
        0xc4, 0x9e, 0x26, 0x78, 0x8e, 0xd0, 0x4a, 0x5d,
        0x60, 0x6c, 0xb4, 0x18, 0xde, 0x74, 0x87, 0x8b,
        0x9a, 0x22, 0xf8, 0xef, 0x29, 0x59, 0x0b, 0xc4,
        0xeb, 0x57, 0xc9, 0xfa, 0xf7, 0xc4, 0x15, 0x24,
        0xa8, 0x85, 0xb8, 0x97, 0x9c, 0x42, 0x3f, 0x2f,
        0x8f, 0x8e, 0x05, 0x92, 0xa9, 0x87, 0x92, 0x01,
        0xbe, 0x7f, 0xf9, 0x77, 0x7a, 0x16, 0x2a, 0xb8,
        0x10, 0xfe, 0xb3, 0x24, 0xba, 0x74, 0xc4, 0xc1,
        0x56, 0xe0, 0x4d, 0x39, 0x09, 0x72, 0x09, 0x65,
        0x3a, 0xc3, 0x3e, 0x5a, 0x5f, 0x2d, 0x88, 0x67
};

static const uint8_t T4_CTR_BIT[] = { 0 };
static const uint8_t A4_CTR_BIT[] = { 0 };
#define A4_CTR_BIT_len 0
#define P4_CTR_BIT_len 1022

static const uint8_t K5_CTR_BIT[] = {
        0x96, 0x18, 0xae, 0x46, 0x89, 0x1f, 0x86, 0x57,
        0x8e, 0xeb, 0xe9, 0x0e, 0xf7, 0xa1, 0x20, 0x2e
};

static const uint8_t IV5_CTR_BIT[] = {
        0xc6, 0x75, 0xa6, 0x4b, 0x64, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Last 3 bits (not to be ciphered) set to 1, since
 * the output buffer is set to all 0xff */
static const uint8_t P5_CTR_BIT[] = {
        0x8d, 0xaa, 0x17, 0xb1, 0xae, 0x05, 0x05, 0x29,
        0xc6, 0x82, 0x7f, 0x28, 0xc0, 0xef, 0x6a, 0x12,
        0x42, 0xe9, 0x3f, 0x8b, 0x31, 0x4f, 0xb1, 0x8a,
        0x77, 0xf7, 0x90, 0xae, 0x04, 0x9f, 0xed, 0xd6,
        0x12, 0x26, 0x7f, 0xec, 0xae, 0xfc, 0x45, 0x01,
        0x74, 0xd7, 0x6d, 0x9f, 0x9a, 0xa7, 0x75, 0x5a,
        0x30, 0xcd, 0x90, 0xa9, 0xa5, 0x87, 0x4b, 0xf4,
        0x8e, 0xaf, 0x70, 0xee, 0xa3, 0xa6, 0x2a, 0x25,
        0x0a, 0x8b, 0x6b, 0xd8, 0xd9, 0xb0, 0x8b, 0x08,
        0xd6, 0x4e, 0x32, 0xd1, 0x81, 0x77, 0x77, 0xfb,
        0x54, 0x4d, 0x49, 0xcd, 0x49, 0x72, 0x0e, 0x21,
        0x9d, 0xbf, 0x8b, 0xbe, 0xd3, 0x39, 0x04, 0xe1,
        0xfd, 0x40, 0xa4, 0x1d, 0x37, 0x0a, 0x1f, 0x65,
        0x74, 0x50, 0x95, 0x68, 0x7d, 0x47, 0xba, 0x1d,
        0x36, 0xd2, 0x34, 0x9e, 0x23, 0xf6, 0x44, 0x39,
        0x2c, 0x8e, 0xa9, 0xc4, 0x9d, 0x40, 0xc1, 0x32,
        0x71, 0xaf, 0xf2, 0x64, 0xd0, 0xf2, 0x48, 0x41,
        0xd6, 0x46, 0x5f, 0x09, 0x96, 0xff, 0x84, 0xe6,
        0x5f, 0xc5, 0x17, 0xc5, 0x3e, 0xfc, 0x33, 0x63,
        0xc3, 0x84, 0x92, 0xaf
};

static const uint8_t C5_CTR_BIT[] = {
        0x91, 0x9c, 0x8c, 0x33, 0xd6, 0x67, 0x89, 0x70,
        0x3d, 0x05, 0xa0, 0xd7, 0xce, 0x82, 0xa2, 0xae,
        0xac, 0x4e, 0xe7, 0x6c, 0x0f, 0x4d, 0xa0, 0x50,
        0x33, 0x5e, 0x8a, 0x84, 0xe7, 0x89, 0x7b, 0xa5,
        0xdf, 0x2f, 0x36, 0xbd, 0x51, 0x3e, 0x3d, 0x0c,
        0x85, 0x78, 0xc7, 0xa0, 0xfc, 0xf0, 0x43, 0xe0,
        0x3a, 0xa3, 0xa3, 0x9f, 0xba, 0xad, 0x7d, 0x15,
        0xbe, 0x07, 0x4f, 0xaa, 0x5d, 0x90, 0x29, 0xf7,
        0x1f, 0xb4, 0x57, 0xb6, 0x47, 0x83, 0x47, 0x14,
        0xb0, 0xe1, 0x8f, 0x11, 0x7f, 0xca, 0x10, 0x67,
        0x79, 0x45, 0x09, 0x6c, 0x8c, 0x5f, 0x32, 0x6b,
        0xa8, 0xd6, 0x09, 0x5e, 0xb2, 0x9c, 0x3e, 0x36,
        0xcf, 0x24, 0x5d, 0x16, 0x22, 0xaa, 0xfe, 0x92,
        0x1f, 0x75, 0x66, 0xc4, 0xf5, 0xd6, 0x44, 0xf2,
        0xf1, 0xfc, 0x0e, 0xc6, 0x84, 0xdd, 0xb2, 0x13,
        0x49, 0x74, 0x76, 0x22, 0xe2, 0x09, 0x29, 0x5d,
        0x27, 0xff, 0x3f, 0x95, 0x62, 0x33, 0x71, 0xd4,
        0x9b, 0x14, 0x7c, 0x0a, 0xf4, 0x86, 0x17, 0x1f,
        0x22, 0xcd, 0x04, 0xb1, 0xcb, 0xeb, 0x26, 0x58,
        0x22, 0x3e, 0x69, 0x3f
};

static const uint8_t T5_CTR_BIT[] = { 0 };
static const uint8_t A5_CTR_BIT[] = { 0 };
#define A5_CTR_BIT_len 0
#define P5_CTR_BIT_len 1245

static const uint8_t K6_CTR_BIT[] = {
        0x54, 0xf4, 0xe2, 0xe0, 0x4c, 0x83, 0x78, 0x6e,
        0xec, 0x8f, 0xb5, 0xab, 0xe8, 0xe3, 0x65, 0x66

};
static const uint8_t IV6_CTR_BIT[] = {
        0xac, 0xa4, 0xf5, 0x0f, 0x58, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Last 3 bits (not to be ciphered) set to 1, since
 * the output buffer is set to all 0xff */
static const uint8_t P6_CTR_BIT[] = {
        0x40, 0x98, 0x1b, 0xa6, 0x82, 0x4c, 0x1b, 0xfb,
        0x42, 0x86, 0xb2, 0x99, 0x78, 0x3d, 0xaf, 0x44,
        0x2c, 0x09, 0x9f, 0x7a, 0xb0, 0xf5, 0x8d, 0x5c,
        0x8e, 0x46, 0xb1, 0x04, 0xf0, 0x8f, 0x01, 0xb4,
        0x1a, 0xb4, 0x85, 0x47, 0x20, 0x29, 0xb7, 0x1d,
        0x36, 0xbd, 0x1a, 0x3d, 0x90, 0xdc, 0x3a, 0x41,
        0xb4, 0x6d, 0x51, 0x67, 0x2a, 0xc4, 0xc9, 0x66,
        0x3a, 0x2b, 0xe0, 0x63, 0xda, 0x4b, 0xc8, 0xd2,
        0x80, 0x8c, 0xe3, 0x3e, 0x2c, 0xcc, 0xbf, 0xc6,
        0x34, 0xe1, 0xb2, 0x59, 0x06, 0x08, 0x76, 0xa0,
        0xfb, 0xb5, 0xa4, 0x37, 0xeb, 0xcc, 0x8d, 0x31,
        0xc1, 0x9e, 0x44, 0x54, 0x31, 0x87, 0x45, 0xe3,
        0xfa, 0x16, 0xbb, 0x11, 0xad, 0xae, 0x24, 0x88,
        0x79, 0xfe, 0x52, 0xdb, 0x25, 0x43, 0xe5, 0x3c,
        0xf4, 0x45, 0xd3, 0xd8, 0x28, 0xce, 0x0b, 0xf5,
        0xc5, 0x60, 0x59, 0x3d, 0x97, 0x27, 0x8a, 0x59,
        0x76, 0x2d, 0xd0, 0xc2, 0xc9, 0xcd, 0x68, 0xd4,
        0x49, 0x6a, 0x79, 0x25, 0x08, 0x61, 0x40, 0x14,
        0xb1, 0x3b, 0x6a, 0xa5, 0x11, 0x28, 0xc1, 0x8c,
        0xd6, 0xa9, 0x0b, 0x87, 0x97, 0x8c, 0x2f, 0xf1,
        0xca, 0xbe, 0x7d, 0x9f, 0x89, 0x8a, 0x41, 0x1b,
        0xfd, 0xb8, 0x4f, 0x68, 0xf6, 0x72, 0x7b, 0x14,
        0x99, 0xcd, 0xd3, 0x0d, 0xf0, 0x44, 0x3a, 0xb4,
        0xa6, 0x66, 0x53, 0x33, 0x0b, 0xcb, 0xa1, 0x10,
        0x5e, 0x4c, 0xec, 0x03, 0x4c, 0x73, 0xe6, 0x05,
        0xb4, 0x31, 0x0e, 0xaa, 0xad, 0xcf, 0xd5, 0xb0,
        0xca, 0x27, 0xff, 0xd8, 0x9d, 0x14, 0x4d, 0xf4,
        0x79, 0x27, 0x59, 0x42, 0x7c, 0x9c, 0xc1, 0xf8,
        0xcd, 0x8c, 0x87, 0x20, 0x23, 0x64, 0xb8, 0xa6,
        0x87, 0x95, 0x4c, 0xb0, 0x5a, 0x8d, 0x4e, 0x2d,
        0x99, 0xe7, 0x3d, 0xb1, 0x60, 0xde, 0xb1, 0x80,
        0xad, 0x08, 0x41, 0xe9, 0x67, 0x41, 0xa5, 0xd5,
        0x9f, 0xe4, 0x18, 0x9f, 0x15, 0x42, 0x00, 0x26,
        0xfe, 0x4c, 0xd1, 0x21, 0x04, 0x93, 0x2f, 0xb3,
        0x8f, 0x73, 0x53, 0x40, 0x43, 0x8a, 0xaf, 0x7e,
        0xca, 0x6f, 0xd5, 0xcf, 0xd3, 0xa1, 0x95, 0xce,
        0x5a, 0xbe, 0x65, 0x27, 0x2a, 0xf6, 0x07, 0xad,
        0xa1, 0xbe, 0x65, 0xa6, 0xb4, 0xc9, 0xc0, 0x69,
        0x32, 0x34, 0x09, 0x2c, 0x4d, 0x01, 0x8f, 0x17,
        0x56, 0xc6, 0xdb, 0x9d, 0xc8, 0xa6, 0xd8, 0x0b,
        0x88, 0x81, 0x38, 0x61, 0x6b, 0x68, 0x12, 0x62,
        0xf9, 0x54, 0xd0, 0xe7, 0x71, 0x17, 0x48, 0x78,
        0x0d, 0x92, 0x29, 0x1d, 0x86, 0x29, 0x99, 0x72,
        0xdb, 0x74, 0x1c, 0xfa, 0x4f, 0x37, 0xb8, 0xb5,
        0x6c, 0xdb, 0x18, 0xa7, 0xca, 0x82, 0x18, 0xe8,
        0x6e, 0x4b, 0x4b, 0x71, 0x6a, 0x4d, 0x04, 0x37,
        0x1f, 0xbe, 0xc2, 0x62, 0xfc, 0x5a, 0xd0, 0xb3,
        0x81, 0x9b, 0x18, 0x7b, 0x97, 0xe5, 0x5b, 0x1a,
        0x4d, 0x7c, 0x19, 0xee, 0x24, 0xc8, 0xb4, 0xd7,
        0x72, 0x3c, 0xfe, 0xdf, 0x04, 0x5b, 0x8a, 0xca,
        0xe4, 0x86, 0x95, 0x17, 0xd8, 0x0e, 0x50, 0x61,
        0x5d, 0x90, 0x35, 0xd5, 0xd9, 0xc5, 0xa4, 0x0a,
        0xf6, 0x02, 0x28, 0x0b, 0x54, 0x25, 0x97, 0xb0,
        0xcb, 0x18, 0x61, 0x9e, 0xeb, 0x35, 0x92, 0x57,
        0x59, 0xd1, 0x95, 0xe1, 0x00, 0xe8, 0xe4, 0xaa,
        0x0c, 0x38, 0xa3, 0xc2, 0xab, 0xe0, 0xf3, 0xd8,
        0xff, 0x04, 0xf3, 0xc3, 0x3c, 0x29, 0x50, 0x69,
        0xc2, 0x36, 0x94, 0xb5, 0xbb, 0xea, 0xcd, 0xd5,
        0x42, 0xe2, 0x8e, 0x8a, 0x94, 0xed, 0xb9, 0x11,
        0x9f, 0x41, 0x2d, 0x05, 0x4b, 0xe1, 0xfa, 0x72,
        0x00, 0xb0, 0x97
};

static const uint8_t C6_CTR_BIT[] = {
        0x5c, 0xb7, 0x2c, 0x6e, 0xdc, 0x87, 0x8f, 0x15,
        0x66, 0xe1, 0x02, 0x53, 0xaf, 0xc3, 0x64, 0xc9,
        0xfa, 0x54, 0x0d, 0x91, 0x4d, 0xb9, 0x4c, 0xbe,
        0xe2, 0x75, 0xd0, 0x91, 0x7c, 0xa6, 0xaf, 0x0d,
        0x77, 0xac, 0xb4, 0xef, 0x3b, 0xbe, 0x1a, 0x72,
        0x2b, 0x2e, 0xf5, 0xbd, 0x1d, 0x4b, 0x8e, 0x2a,
        0xa5, 0x02, 0x4e, 0xc1, 0x38, 0x8a, 0x20, 0x1e,
        0x7b, 0xce, 0x79, 0x20, 0xae, 0xc6, 0x15, 0x89,
        0x5f, 0x76, 0x3a, 0x55, 0x64, 0xdc, 0xc4, 0xc4,
        0x82, 0xa2, 0xee, 0x1d, 0x8b, 0xfe, 0xcc, 0x44,
        0x98, 0xec, 0xa8, 0x3f, 0xbb, 0x75, 0xf9, 0xab,
        0x53, 0x0e, 0x0d, 0xaf, 0xbe, 0xde, 0x2f, 0xa5,
        0x89, 0x5b, 0x82, 0x99, 0x1b, 0x62, 0x77, 0xc5,
        0x29, 0xe0, 0xf2, 0x52, 0x9d, 0x7f, 0x79, 0x60,
        0x6b, 0xe9, 0x67, 0x06, 0x29, 0x6d, 0xed, 0xfa,
        0x9d, 0x74, 0x12, 0xb6, 0x16, 0x95, 0x8c, 0xb5,
        0x63, 0xc6, 0x78, 0xc0, 0x28, 0x25, 0xc3, 0x0d,
        0x0a, 0xee, 0x77, 0xc4, 0xc1, 0x46, 0xd2, 0x76,
        0x54, 0x12, 0x42, 0x1a, 0x80, 0x8d, 0x13, 0xce,
        0xc8, 0x19, 0x69, 0x4c, 0x75, 0xad, 0x57, 0x2e,
        0x9b, 0x97, 0x3d, 0x94, 0x8b, 0x81, 0xa9, 0x33,
        0x7c, 0x3b, 0x2a, 0x17, 0x19, 0x2e, 0x22, 0xc2,
        0x06, 0x9f, 0x7e, 0xd1, 0x16, 0x2a, 0xf4, 0x4c,
        0xde, 0xa8, 0x17, 0x60, 0x36, 0x65, 0xe8, 0x07,
        0xce, 0x40, 0xc8, 0xe0, 0xdd, 0x9d, 0x63, 0x94,
        0xdc, 0x6e, 0x31, 0x15, 0x3f, 0xe1, 0x95, 0x5c,
        0x47, 0xaf, 0xb5, 0x1f, 0x26, 0x17, 0xee, 0x0c,
        0x5e, 0x3b, 0x8e, 0xf1, 0xad, 0x75, 0x74, 0xed,
        0x34, 0x3e, 0xdc, 0x27, 0x43, 0xcc, 0x94, 0xc9,
        0x90, 0xe1, 0xf1, 0xfd, 0x26, 0x42, 0x53, 0xc1,
        0x78, 0xde, 0xa7, 0x39, 0xc0, 0xbe, 0xfe, 0xeb,
        0xcd, 0x9f, 0x9b, 0x76, 0xd4, 0x9c, 0x10, 0x15,
        0xc9, 0xfe, 0xcf, 0x50, 0xe5, 0x3b, 0x8b, 0x52,
        0x04, 0xdb, 0xcd, 0x3e, 0xed, 0x86, 0x38, 0x55,
        0xda, 0xbc, 0xdc, 0xc9, 0x4b, 0x31, 0xe3, 0x18,
        0x02, 0x15, 0x68, 0x85, 0x5c, 0x8b, 0x9e, 0x52,
        0xa9, 0x81, 0x95, 0x7a, 0x11, 0x28, 0x27, 0xf9,
        0x78, 0xba, 0x96, 0x0f, 0x14, 0x47, 0x91, 0x1b,
        0x31, 0x7b, 0x55, 0x11, 0xfb, 0xcc, 0x7f, 0xb1,
        0x3a, 0xc1, 0x53, 0xdb, 0x74, 0x25, 0x11, 0x17,
        0xe4, 0x86, 0x1e, 0xb9, 0xe8, 0x3b, 0xff, 0xff,
        0xc4, 0xeb, 0x77, 0x55, 0x57, 0x90, 0x38, 0xe5,
        0x79, 0x24, 0xb1, 0xf7, 0x8b, 0x3e, 0x1a, 0xd9,
        0x0b, 0xab, 0x2a, 0x07, 0x87, 0x1b, 0x72, 0xdb,
        0x5e, 0xef, 0x96, 0xc3, 0x34, 0x04, 0x49, 0x66,
        0xdb, 0x0c, 0x37, 0xca, 0xfd, 0x1a, 0x89, 0xe5,
        0x64, 0x6a, 0x35, 0x80, 0xeb, 0x64, 0x65, 0xf1,
        0x21, 0xdc, 0xe9, 0xcb, 0x88, 0xd8, 0x5b, 0x96,
        0xcf, 0x23, 0xcc, 0xcc, 0xd4, 0x28, 0x07, 0x67,
        0xbe, 0xe8, 0xee, 0xb2, 0x3d, 0x86, 0x52, 0x46,
        0x1d, 0xb6, 0x49, 0x31, 0x03, 0x00, 0x3b, 0xaf,
        0x89, 0xf5, 0xe1, 0x82, 0x61, 0xea, 0x43, 0xc8,
        0x4a, 0x92, 0xeb, 0xff, 0xff, 0xe4, 0x90, 0x9d,
        0xc4, 0x6c, 0x51, 0x92, 0xf8, 0x25, 0xf7, 0x70,
        0x60, 0x0b, 0x96, 0x02, 0xc5, 0x57, 0xb5, 0xf8,
        0xb4, 0x31, 0xa7, 0x9d, 0x45, 0x97, 0x7d, 0xd9,
        0xc4, 0x1b, 0x86, 0x3d, 0xa9, 0xe1, 0x42, 0xe9,
        0x00, 0x20, 0xcf, 0xd0, 0x74, 0xd6, 0x92, 0x7b,
        0x7a, 0xb3, 0xb6, 0x72, 0x5d, 0x1a, 0x6f, 0x3f,
        0x98, 0xb9, 0xc9, 0xda, 0xa8, 0x98, 0x2a, 0xff,
        0x06, 0x78, 0x2f
};

static const uint8_t T6_CTR_BIT[] = { 0 };
static const uint8_t A6_CTR_BIT[] = { 0 };
#define A6_CTR_BIT_len 0
#define P6_CTR_BIT_len 3861

static const struct gcm_ctr_vector ctr_vectors[] = {
	/*
         * field order {K, Klen, IV, IVlen, A, Alen, P, Plen, C, T, Tlen};
         * original vector does not have a valid sub hash key
         */
	vector(1_CTR),
	vector(2_CTR),
	vector(3_CTR),
	vector(4_CTR),
	vector(5_CTR),
	vector(6_CTR),
	vector(7_CTR),
	vector(8_CTR),
	vector(9_CTR),
	vector(10_CTR),
	vector(11_CTR),
	vector(12_CTR),
	vector(13_CTR),
	vector(14_CTR),
	vector(15_CTR)
};

static const struct gcm_ctr_vector ctr_bit_vectors[] = {
	bit_vector(1_CTR_BIT),
	bit_vector(2_CTR_BIT),
	bit_vector(3_CTR_BIT),
	bit_vector(4_CTR_BIT),
	bit_vector(5_CTR_BIT),
	bit_vector(6_CTR_BIT)
};

static int
test_ctr(struct MB_MGR *mb_mgr,
         const void *expkey,
         unsigned key_len,
         const void *iv,
         unsigned iv_len,
         const uint8_t *in_text,
         const uint8_t *out_text,
         unsigned text_len,
         int dir,
         int order,
         const JOB_CIPHER_MODE alg)
{
        uint32_t text_byte_len;
        struct JOB_AES_HMAC *job;
        uint8_t padding[16];
        uint8_t *target;
        int ret = -1;

        /* Get number of bytes (in case algo is CNTR_BITLEN) */
        if (alg == CNTR)
                text_byte_len = text_len;
        else
                text_byte_len = BYTE_ROUND_UP(text_len);

        target = malloc(text_byte_len + (sizeof(padding) * 2));
        if (target == NULL) {
		fprintf(stderr, "Can't allocate buffer memory\n");
		goto end;
        }

        memset(target, -1, text_byte_len + (sizeof(padding) * 2));
        memset(padding, -1, sizeof(padding));

        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL)
                ;

        job = IMB_GET_NEXT_JOB(mb_mgr);
        job->cipher_direction = dir;
        job->chain_order = order;
        job->dst = target + 16;
        job->src = in_text;
        job->cipher_mode = alg;
        job->aes_enc_key_expanded = expkey;
        job->aes_dec_key_expanded = expkey;
        job->aes_key_len_in_bytes = key_len;
        job->iv = iv;
        job->iv_len_in_bytes = iv_len;
        job->cipher_start_src_offset_in_bytes = 0;
        if (alg == CNTR)
                job->msg_len_to_cipher_in_bytes = text_byte_len;
        else
                job->msg_len_to_cipher_in_bits = text_len;

        job->hash_alg = NULL_HASH;

        job = IMB_SUBMIT_JOB(mb_mgr);
        if (!job) {
                printf("%d Unexpected null return from submit_job\n", __LINE__);
                goto end;
        }
        if (job->status != STS_COMPLETED) {
                printf("%d Error status:%d", __LINE__, job->status);
                goto end;
        }
        job = IMB_FLUSH_JOB(mb_mgr);
        if (job) {
                printf("%u Unexpected return from flush_job\n", __LINE__);
                goto end;
        }

        if (memcmp(out_text, target + 16, text_byte_len)) {
                printf("mismatched\n");
                hexdump(stderr, "Target", target, text_byte_len + 32);
                hexdump(stderr, "Expected", out_text, text_byte_len);
                goto end;
        }
        if (memcmp(padding, target, sizeof(padding))) {
                printf("overwrite head\n");
                hexdump(stderr, "Target", target, text_byte_len + 32);
                goto end;
        }
        if (memcmp(padding, target + sizeof(padding) + text_byte_len,
                   sizeof(padding))) {
                printf("overwrite tail\n");
                hexdump(stderr, "Target", target, text_byte_len + 32);
                goto end;
        }
        ret = 0;
        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL)
                ;
 end:
        if (target != NULL)
                free(target);
        return ret;
}

static int
test_ctr_vectors(struct MB_MGR *mb_mgr, const struct gcm_ctr_vector *vectors,
                 const uint32_t vectors_cnt, const JOB_CIPHER_MODE alg)
{
	uint32_t vect;
	int errors = 0;
        DECLARE_ALIGNED(uint32_t expkey[4*15], 16);
        DECLARE_ALIGNED(uint32_t dust[4*15], 16);

	printf("AES-CTR standard test vectors:\n");
	for (vect = 0; vect < vectors_cnt; vect++) {
#ifdef DEBUG
        if (alg == CNTR)
		printf("Standard vector %d/%d  Keylen:%d IVlen:%d PTLen:%d ",
                       vect, vectors_cnt - 1,
                       (int) vectors[vect].Klen,
                       (int) vectors[vect].IVlen,
                       (int) vectors[vect].Plen);
        else
		printf("Bit vector %d/%d  Keylen:%d IVlen:%d PTLen:%d ",
                       vect, vectors_cnt - 1,
                       (int) vectors[vect].Klen,
                       (int) vectors[vect].IVlen,
                       (int) vectors[vect].Plen);
#else
		printf(".");
#endif


                switch (vectors[vect].Klen) {
                case BITS_128:
                        IMB_AES_KEYEXP_128(mb_mgr, vectors[vect].K,
                                           expkey, dust);
                        break;
                case BITS_192:
                        IMB_AES_KEYEXP_192(mb_mgr, vectors[vect].K,
                                           expkey, dust);
                        break;
                case BITS_256:
                        IMB_AES_KEYEXP_256(mb_mgr, vectors[vect].K,
                                           expkey, dust);
                        break;
                default:
                        return -1;
                }

                if (test_ctr(mb_mgr,
                             expkey, vectors[vect].Klen,
                             vectors[vect].IV,
                             (unsigned) vectors[vect].IVlen,
                             vectors[vect].P, vectors[vect].C,
                             (unsigned) vectors[vect].Plen,
                             ENCRYPT, CIPHER_HASH, alg)) {
                        printf("error #%d encrypt\n", vect + 1);
                        errors++;
                }

                if (test_ctr(mb_mgr,
                             expkey, vectors[vect].Klen,
                             vectors[vect].IV,
                             (unsigned) vectors[vect].IVlen,
                             vectors[vect].C, vectors[vect].P,
                             (unsigned) vectors[vect].Plen,
                             DECRYPT, HASH_CIPHER, alg)) {
                        printf("error #%d decrypt\n", vect + 1);
                        errors++;
                }

                if (vectors[vect].IVlen == 12) {
                        /* IV in the table didn't
                         * include block counter (12 bytes).
                         * Let's encrypt & decrypt the same but
                         * with 16 byte IV that includes block counter.
                         */
                        const unsigned new_iv_len = 16;
                        const unsigned orig_iv_len = 12;
                        uint8_t local_iv[16];

                        memcpy(local_iv, vectors[vect].IV, orig_iv_len);
                        /* 32-bit 0x01000000 in LE */
                        local_iv[12] = 0x00;
                        local_iv[13] = 0x00;
                        local_iv[14] = 0x00;
                        local_iv[15] = 0x01;

                        if (test_ctr(mb_mgr,
                                     expkey, vectors[vect].Klen,
                                     local_iv, new_iv_len,
                                     vectors[vect].P, vectors[vect].C,
                                     (unsigned) vectors[vect].Plen,
                                     ENCRYPT, CIPHER_HASH, alg)) {
                                printf("error #%d encrypt\n", vect + 1);
                                errors++;
                        }

                        if (test_ctr(mb_mgr,
                                     expkey, vectors[vect].Klen,
                                     local_iv, new_iv_len,
                                     vectors[vect].C, vectors[vect].P,
                                     (unsigned) vectors[vect].Plen,
                                     DECRYPT, HASH_CIPHER, alg)) {
                                printf("error #%d decrypt\n", vect + 1);
                                errors++;
                        }
                }
	}
	printf("\n");
	return errors;
}

int
ctr_test(const enum arch_type arch,
         struct MB_MGR *mb_mgr)
{
        int errors;

        (void) arch; /* unused */

	const uint32_t ctr_vec_cnt = DIM(ctr_vectors);
	const uint32_t ctr_bit_vec_cnt = DIM(ctr_bit_vectors);

        /* Standard CTR vectors */
        errors = test_ctr_vectors(mb_mgr, ctr_vectors, ctr_vec_cnt, CNTR);

        /* CTR_BITLEN vectors */
        errors += test_ctr_vectors(mb_mgr, ctr_bit_vectors, ctr_bit_vec_cnt,
                                   CNTR_BITLEN);

	if (0 == errors)
		printf("...Pass\n");
	else
		printf("...Fail\n");

	return errors;
}
