/*
 * Copyright (C) 2020 Intel Corporation
 *
 * SPDX-License-Identifier: MIT
 *
 */

#include <cstdint>

#ifdef _WIN32
#define SIGNATURE __declspec(dllexport) int __cdecl
#else
#define SIGNATURE int
#endif

extern "C" {
/// Invokes ocloc API using C interface. Supported commands match
/// the functionality of ocloc executable (check ocloc's "help"
/// for reference : shared/offline_compiler/source/ocloc_api.cpp).
///
/// numArgs and argv params represent the command line.
/// Remaining params represent I/O.
/// Output params should be freed using oclocFreeOutput when
/// no longer needed.
/// List and names of outputs match outputs of ocloc executable.
///
/// \param numArgs is the number of arguments to pass to ocloc
///
/// \param argv is an array of arguments to be passed to ocloc
///
/// \param numSources is the number of in-memory representations
/// of source files to be passed to ocloc
///
/// \param dataSources is an array of in-memory representations
/// of source files to be passed to ocloc
///
/// \param lenSources is an array of sizes of in-memory representations
/// of source files passed to ocloc as dataSources
///
/// \param nameSources is an array of names of in-memory representations
/// of source files passed to ocloc as dataSources
///
/// \param numInputHeaders is the number of in-memory representations
/// of header files to be passed to ocloc
///
/// \param dataInputHeaders is an array of in-memory representations
/// of header files to be passed to ocloc
///
/// \param lenInputHeaders is an array of sizes of in-memory representations
/// of header files passed to ocloc as dataInputHeaders
///
/// \param nameInputHeaders is an array of names of in-memory representations
/// of header files passed to ocloc as dataInputHeaders
///
/// \param numOutputs returns the number of outputs
///
/// \param dataOutputs returns an array of in-memory representations
/// of output files
///
/// \param lenOutputs returns an array of sizes of in-memory representations
/// of output files
///
/// \param nameOutputs returns an array of names of in-memory representations
/// of output files. Special name stdout.log describes output that contains
/// messages generated by ocloc (e.g. compiler errors/warnings)
///
/// \returns 0 on succes. Returns non-0 in case of failure.
SIGNATURE oclocInvoke(unsigned int numArgs, const char *argv[],
                      const uint32_t numSources, const uint8_t **dataSources, const uint64_t *lenSources, const char **nameSources,
                      const uint32_t numInputHeaders, const uint8_t **dataInputHeaders, const uint64_t *lenInputHeaders, const char **nameInputHeaders,
                      uint32_t *numOutputs, uint8_t ***dataOutputs, uint64_t **lenOutputs, char ***nameOutputs);

/// Frees results of oclocInvoke
///
/// \param numOutputs is number of outputs as returned by oclocInvoke
///
/// \param dataOutputs is array of outputs as returned by oclocInvoke
///
/// \param lenOutputs is array of sizes of outputs as returned by oclocInvoke
///
/// \param nameOutputs is array of names of outputs as returned by oclocInvoke
///
/// \returns 0 on succes. Returns non-0 in case of failure.
SIGNATURE oclocFreeOutput(uint32_t *numOutputs, uint8_t ***dataOutputs, uint64_t **lenOutputs, char ***nameOutputs);
}
