/* NVTV mmio/port  wrapper -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: mmio.c,v 1.3 2004/03/04 09:12:56 dthierbach Exp $
 *
 * Contents:
 *
 * Wrapper for mmio/port access for different OS
 *
 */

#include "local.h" /* before everything else */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IO_H
#include <sys/io.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif


#ifdef __FreeBSD__

/* FIXME: Are these and sys/stat.h really needed? */

#include <fcntl.h>
#include <errno.h>

/* FreeBSD does not have ioperm() and iopl() but can easily emulate them */

#define ioperm i386_set_ioperm

/* /dev/io is a controlled security hole. Any FreeBSD process that holds a
 * file descriptor on /dev/io open will get its IOPL bits in the flag
 * register set, thus allowing it to perform direct I/O operations.
 */

static int
iopl (int level)
{
  static int desc = -1;

  if (desc == -1 && level > 0) {
    if ((desc = open ("/dev/io", O_RDWR)) == -1) return EPERM;
  }
  return 0;
}

#endif /* __FreeBSD__ */

#ifdef __NetBSD__
static int
ioperm(from, num, turn_on)
	unsigned long from;
	unsigned long num;
	int turn_on;
{
	u_long *bitmap, mask, value;
	int r = -1;
	int first_index, last_index, i;

	if((bitmap = (u_long *)malloc(1024*sizeof(u_long))) == NULL)
		return -1;

	if(i386_get_ioperm(bitmap) != 0)
		goto ioperm_fail;

	first_index = from / 32;
	last_index = (from + num) / 32;

	/* First set */
	mask = 0xffffffff << (from % 32);
	if(turn_on)
		bitmap[first_index] |= mask;
	else
		bitmap[first_index] &= ~mask;

	/* Last set */
	mask = 0xffffffff >> (31 - (from + num) % 32);
	if(turn_on)
		bitmap[last_index] |= mask;
	else
		bitmap[last_index] &= ~mask;

	if(last_index > first_index + 1) {
		/* Complete sets */
		value = turn_on ? 0xffffffff : 0;

		for(i = first_index+1; i<last_index; i++)
			bitmap[i] = value;
	}

	r = i386_set_ioperm(bitmap);

ioperm_fail:
	free(bitmap);
	return r;
}
#endif /* __NetBSD__ */

#ifdef HAVE_WINIO

void 
mmio_port_perm (unsigned long from, unsigned long num, int turn_on)
{
  /* FIXME */
}

#else /* HAVE_WINIO */

void 
mmio_port_perm (unsigned long from, unsigned long num, int turn_on)
{
  if (from + num >= 0x400) {
#ifdef __NetBSD__
    /* FIXME: Limit is 0x8000 for NetBSD ?? */
    if (turn_on) i386_iopl (3); else i386_iopl (0);
#else
    if (turn_on) iopl (3); else iopl (0);
#endif
  } else {
    ioperm (from, num, turn_on);
  }
}

#endif /* HAVE_WINIO */
