// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

// ************************************************************************* //
//                          avtExtrudedVolWriter.C                           //
// ************************************************************************* //

#include <avtExtrudedVolWriter.h>

#include <vector>

#include <vtkCellData.h>
#include <vtkDataArray.h>
#include <vtkDataSet.h>
#include <vtkDataSetWriter.h>
#include <vtkPointData.h>

#include <avtDatabaseMetaData.h>
#include <DBOptionsAttributes.h>

#include <InvalidVariableException.h>


using     std::string;
using     std::vector;


// ****************************************************************************
//  Method: avtExtrudedVolWriter constructor
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

avtExtrudedVolWriter::avtExtrudedVolWriter(DBOptionsAttributes *atts)
{
    timestep   = atts->GetInt("Time");
    nTimesteps = atts->GetInt("nTimes");
    numChunks  = -1;
}

// ****************************************************************************
//  Method: avtExtrudedVolWriter::OpenFile
//
//  Purpose:
//      Does no actual work.  Just records the stem name for the files.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::OpenFile(const string &stemname, int numblocks)
{
    stem      = stemname;
    numChunks = numblocks;
}


// ****************************************************************************
//  Method: avtExtrudedVolWriter::WriteHeaders
//
//  Purpose:
//      Writes out a VisIt file to tie the ExtrudedVol files together.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::WriteHeaders(const avtDatabaseMetaData *md,
                           const vector<string> &scalars,
                           const vector<string> &vectors,
                           const vector<string> &materials)
{
    if (timestep == 0)
    {
        char filename[1024];
        snprintf(filename, 1024, "%s.exvol", stem.c_str());

        ofstream ofile(filename);
        if (ofile.fail())
        {
            EXCEPTION1(VisItException, "Unable to open a file for writing "
                                     "when saving out to ExtrudedVol format.");
        }

        ofile << "STEM: " << stem.c_str() << endl;
        ofile << "NUMCHUNKS: " << numChunks << endl;
        ofile << "NTIMES: " << nTimesteps << endl;
        variables = scalars;
        ofile << "VARIABLES: " << variables.size() << endl;
        for (size_t i = 0 ; i < variables.size() ; i++)
            if (strncmp(variables[i].c_str(), "AVT_TRACE_HIST_", 
                        strlen("AVT_TRACE_HIST_")) == 0)
                ofile << variables[i].c_str() + strlen("AVT_TRACE_HIST_")<<endl;
            else
                ofile << variables[i] << endl;
    }
    else
    {
        variables = scalars;
    }
}


// ****************************************************************************
//  Method: avtExtrudedVolWriter::WriteChunk
//
//  Purpose:
//      This writes out one chunk of an avtDataset.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::WriteChunk(vtkDataSet *ds, int chunk)
{
    int  i, j;

    if (timestep == -1)
    {
        vtkDataSetWriter *wrtr = vtkDataSetWriter::New();
        wrtr->SetInputData(ds);
        char filename[1024];
        snprintf(filename, 1024, "%s.%d.exvol_conn", stem.c_str(), chunk);
        wrtr->SetFileName(filename);
        wrtr->Write();
        wrtr->Delete();
    }
    else
    {
        char filename[1024];
        snprintf(filename, 1024, "%s.%d.%d.exvol_var", stem.c_str(), 
                                                       timestep, chunk);

        ofstream ofile(filename);
        if (ofile.fail())
        {
            EXCEPTION1(VisItException, "Unable to open a file for writing "
                                     "when saving out to ExtrudedVol format.");
        }

        int npts = ds->GetNumberOfPoints();
        ofile << npts << endl;

        for (i = 0 ; i < npts ; i++)
        {
            double pt[3];
            ds->GetPoint(i, pt);
            ofile << pt[0] << " " << pt[1] << " " << pt[2] << endl;
        }

        for (i = 0 ; i < (int)variables.size() ; i++)
        {
            vtkDataArray *arr = 
                            ds->GetPointData()->GetArray(variables[i].c_str());
            if (arr == NULL)
                ds->GetCellData()->GetArray(variables[i].c_str());
            if (arr == NULL)
            {
                EXCEPTION1(InvalidVariableException, variables[i].c_str());
            }
            for (j = 0 ; j < npts ; j++)
                ofile << arr->GetTuple1(j) << endl;
        }
    }
}


// ****************************************************************************
//  Method: avtExtrudedVolWriter::CloseFile
//
//  Purpose:
//      Closes the file.  This does nothing in this case.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::CloseFile(void)
{
    ;
}


