/*
 * Copyright (C) 2000-2004 by Oswald Buddenhagen <puf@ossi.cjb.net>
 * based on puf 0.1.x (C) 1999,2000 by Anders Gavare <gavare@hotmail.com>
 *
 * You may modify and distribute this code under the terms of the GPL.
 * There is NO WARRANTY of any kind. See COPYING for details.
 *
 * http_conn.c - prepare creation of a connection
 *
 */

#include "puf.h"

static void
init_aurl(aurl_t *au)
{
    au->f = -1;
    au->size_total = 0;
    au->size_fetched = 0;
    au->buffer = NULL;
    au->offset = 0;
    au->size = 0;
    au->buff = 0;
    au->buff_len = 0;
    au->http_result_code = 0;
    au->http_done_header = 0;
    au->content_is_text = 0;
    au->content_is_html = 0;
    au->reloc = 0;
    au->file_created = 0;
    au->headers = NULL;
    au->hdrslen = au->hdrssiz = 0;

}

/*  find a (working) ip for a host  */
static int 
get_ip(host_t *h, int fail_no_wait)
{
    hinfo_t *hi = h->info;
    int i, j;

    dbg(CON, ("Want to connect '%s' ... ", hi->name));
    if (!hi->num_ips) {
	dbge(CON, ("already given up!\n"));
	return -507 + hi->ips[0].last_errt;
    }
    for (i = j = hi->num_ips; i; i--) {
	if (++hi->cur_ip >= hi->num_ips)
	    hi->cur_ip = 0;
	if (hi->ips[hi->cur_ip].last_errt >= 3)
	    j--;
	else {
	    if (hi->ips[hi->cur_ip].retry_time <= cur_tv.tv_sec) {
		dbge(CON, ("ok\n"));
		return hi->cur_ip;
	    }
	}
    }
    if (fail_no_wait) {
	if (++hi->cur_ip >= hi->num_ips)
	    hi->cur_ip = 0;
	dbge(CON, ("forced\n"));
	return hi->cur_ip;
    }
    dbge(CON, ("\n"));
    if (!j) {
	prx(ERR, "Giving up host '%s'!\n", hi->name);
	hi->num_ips = 0;
	return -507 + hi->ips[0].last_errt;
    }
/*    prx(WRN, "No valid IP address for '%s' by now!\n", hi->name); */
    return -1;
}

#define FAT_BAD_CHARS "\\:*?\"<>|"

/*  prepare url for connection:
    - find working host
    - find working proxy
    - create a "shadow" element
    - calculate a local disposition for the file
*/
int 
activate_url(url_t *u, aurl_t **aup)
{
    const char *disp;
    proxyarr_t *sh;
    struct stat statbuf;
    unsigned dl, wil;		/*  unsigned will catch -1 also  */
    int ipidx, pipidx, i, ql;
    proxy_t *proxy, *prox;
    proxyent_t *proxye, *proxe;
    aurl_t *au;
    off_t foff;
    time_t ftime;
    char buf[SHORTSTR];

    if (u->parm->proxy) {
	proxy = u->parm->proxy;
	if (!proxy->host->ready)
	    return RT_AGAIN;
	if (!proxy->host ||	/* proxy dead (needed auth) */
	    !proxy->host->info)	/* host dead (lookup failed) */
	    goto pfail;
	else {
	    pipidx = get_ip(proxy->host, u->parm->opt->fail_no_wait);
	    if (pipidx < 0) {
		if (pipidx == -1 && u->parm->strictproxy)
		    return RT_AGAIN;
	      pfail:
		if (u->parm->strictproxy)
		    return RT_GIVEUP;
		if (!detach_parm(u))
		    return RT_GIVEUP;
		u->parm->proxy = 0;
		proxy = 0;
		pipidx = 0;
	    }
	}
    } else {
	proxy = 0;
	pipidx = 0;
    }

    proxye = 0;
    sh = u->parm->opt->proxies;
    if (!proxy && sh->nents) {
	int score, tscor, round, rtval;
	u_int pidx;

	if (all_proxy_wait && waiting_proxies)
	    return RT_RETRY;
	/* 
	 * we try the best proxy left in every pass until we find one 
         * that actually works.
         */
	for (round = 0; ; round++) {
	    for (proxy = 0, score = INT_MAX, rtval = RT_GIVEUP,
		 pidx = 0; pidx < sh->nents; pidx++)
	    {
		proxe = sh->ents + pidx;
		prox = proxe->proxy;
		if (!prox->host)		/* proxy dead (needed auth) */
		    continue;
		if (!prox->host->ready) {
		    rtval = RT_RETRY;
		    continue;
		}
		if (!prox->host->info ||	/* host dead (lookup failed) */
		    !prox->host->info->num_ips)	/* host dead (connects failed) */
		    continue;
		if (!round)
		    prox->host->info->maybe = 1;
		else if (!prox->host->info->maybe) {
		    rtval = RT_RETRY;
		    continue;
		}
		tscor = proxe->score * 100 / proxe->ratio;
		if (tscor < score) {
		    score = tscor;
		    proxye = proxe;
		    proxy = prox;
		}
	    }
	    if (!proxy)
		return rtval;
	    pipidx = get_ip(proxy->host, u->parm->opt->fail_no_wait);
	    if (pipidx >= 0)
		break;
	    proxy->host->info->maybe = 0;
	}
    }

    ipidx = get_ip(u->host, u->parm->opt->fail_no_wait);
    if (ipidx == -1)
	return RT_AGAIN;	/*  transient server problem  */
    if (ipidx < 0) {
	uwerrm(u, -ipidx, "!Giving up $u (host given up)");
	return RT_GIVEUP;
    }

    foff = 0;
    ftime = 0;

    dl = 0;

    /*  decide about disk file name for the url  */
	if (u->parm->disposition) {
	  if (u->parm->disposition->disp[0]) {
	    /* lenght checked by adden() */
	    if (u->parm->disposition->disp[0] != '/' &&
		u->parm->opt->disp_path->path[0])
	    {
		cat_str(buf, dl, u->parm->opt->disp_path->path);
		cat_chr(buf, dl, '/');
	    }
	    cat_str(buf, dl, u->parm->disposition->disp);
	  }
	} else {
	    if (u->parm->opt->disp_path->path[0]) {
		cat_str(buf, dl, u->parm->opt->disp_path->path);
		cat_chr(buf, dl, '/');
	    }
	    /* lenght checked by adden() up to here */
	    if (u->parm->opt->dir_mode == DIRS_NONE);
	    else if (u->parm->opt->dir_mode == DIRS_ALWAYS ||
		     u->parm->opt->follows_max >= HOST_RECURSIVE ||
		     (u->parm->opt->follows_max >= SAMEDIR_RECURSIVE && 
		      u->disp_pathoff < 0))
	    {
		lcat_str(buf, sizeof(buf), dl, u->host->info->lname);
		lcat_chr(buf, sizeof(buf), dl, '/');
		lcat_mem(buf, sizeof(buf), dl, u->path_len, u->local_part);
	    }
	    else if (u->parm->opt->follows_max >= SAMEDIR_RECURSIVE)
		lcat_mem(buf, sizeof(buf), dl,
			 u->path_len - u->disp_pathoff,
			 u->local_part + u->disp_pathoff);

	    disp = u->local_part[u->path_len] ?
		u->local_part + u->path_len : 
		u->parm->opt->index_filename ? 
		u->parm->opt->index_filename : DEFAULT_INDEX_FILE_NAME;

	    for (;; disp++) {
		if (dl >= sizeof(buf) - sizeof(PART_EXT))
		    return RT_GIVEUP;
		if (!*disp)
		    break;
		buf[dl++] = *disp == '/' ? '!' : *disp;
	    }
#ifndef HAVE_CYGWIN
	    if (u->parm->opt->fat_quotes)
#endif
	    {
		for (i = dl, ql = sizeof(buf) - sizeof(PART_EXT); --i >= 0; )
		{
		    unsigned char c = buf[i];
		    static const char hextab[] = "0123456789abcdef";
		    if (memchr("#"FAT_BAD_CHARS, c, sizeof(FAT_BAD_CHARS))) {
			if ((ql -= 3) < i)
			    return RT_GIVEUP;
			buf[ql + 2] = hextab[c & 15];
			buf[ql + 1] = hextab[c >> 4];
			buf[ql] = '#';
		    } else {
			if (--ql < i)
			    return RT_GIVEUP;
			buf[ql] = c;
		    }
		}
		dl = sizeof(buf) - sizeof(PART_EXT) - ql;
		memcpy(buf, buf + ql, dl);
	    }
	    buf[dl] = 0;
	}
	wil = dl;

      /*  decide if we want to continue a download or skip the file at all  */
      if (dl && u->parm->opt->update_mode != EX_CLOBBER) {
	if (!stat(buf, &statbuf)) {
	    if (u->parm->opt->update_mode == EX_UPDATE)
		/*  mark it as update canditate  */
		ftime = statbuf.st_mtime;
	    else {		/*  EX_CONTINUE & EX_NO_CLOBBER  */
		if (u->parm->opt->update_mode == EX_NO_CLOBBER)
		    prxu(WRN, u, "$u: file %s exists\n", buf);
		else if (needs_recurse_u(u, 0))
		    /*  scan the file for links. we scan the file with no
		       regard to it's content type - we simply don't know it.
		       extension-based type detection is too unreliable.  */
		    recurse_file(u, buf);
		return RT_SKIP;
	    }
	} else {
	    memcpy(buf + dl, PART_EXT, sizeof(PART_EXT));
	    if (!stat(buf, &statbuf)) {
		if (u->parm->opt->update_mode == EX_NO_CLOBBER) {
		    prxu(WRN, u, "$u: file %s exists\n", buf);
		    return RT_SKIP;
		} else {	/*  EX_UPDATE & EX_CONTINUE  */
		    /*  mark as continuation candidate  */
		    foff = statbuf.st_size;
		    ftime = statbuf.st_mtime;
		}
	    }
	}
      }

    if (u->parm->time_stamp)
	ftime = u->parm->time_stamp;

    if (!(au = mmalloc(sizeof(*au) + (dl ? dl + sizeof(PART_EXT) : 1))))
	return RT_RETRY;

    au->file_off = foff;
    au->file_time = ftime;
    au->url = u;
    au->displen = wil;
  if (dl) {
    memcpy(au->disposition, buf, dl);
    memcpy(au->disposition + dl, PART_EXT, sizeof(PART_EXT));
  } else
    au->disposition[0] = 0;

    init_aurl(au);

    au->auth_chall = 0;

    au->ipidx = ipidx;
    au->pipidx = pipidx;
    au->proxye = proxye;
    au->proxy = proxy;

    dbgu(CON, (u, "activated $u - ipidx: %d  proxy: '%s'  pipidx: %d\n", ipidx, proxy ? proxy->host->name : "(null)", pipidx));

    *aup = au;
    return RT_OK;
}


/*  try to connect to a host. returns -1 on error, otherwise a file
    descriptor number is returned (socket number). */
static int 
tcp_connect(struct in_addr addr, u_short port, struct in_addr baddr)
{
    int s;
    struct sockaddr_in server_in;

    if ((s = socket(PF_INET, SOCK_STREAM, 0)) < 0 && 
	    (!free_fd(0) || (s = socket(PF_INET, SOCK_STREAM, 0)) < 0))
	return -1;

    if (baddr.s_addr) {
	bind_addr.sin_addr = baddr;
	bind(s, (struct sockaddr *)&bind_addr, sizeof(struct sockaddr));
    }

    fcntl(s, F_SETFL, fcntl(s, F_GETFL) | O_NONBLOCK);

    server_in.sin_family = AF_INET;
    server_in.sin_addr = addr;
    server_in.sin_port = htons(port);

    connect(s, (struct sockaddr *)&server_in, sizeof(server_in));

    return s;
}

static void
_deactivate_url(aurl_t *au)
{
    if (au->auth_chall)
	free(au->auth_chall);
    free(au);
}

int
connect_url(aurl_t *au, struct timeval *next_conn_tv)
{
	ptrarr_t *sh;
	struct in_addr addr;
	struct in_addr baddr;
	u_short port;

	if (au->proxy) {
	    addr = au->proxy->host->info->ips[au->pipidx].addr;
	    port = au->proxy->port;
	} else {
	    addr = au->url->host->info->ips[au->ipidx].addr;
	    port = au->url->port;
	}
	sh = au->url->parm->opt->bind_addrs;
	if (sh->nents) {
	    baddr = *((struct in_addr **)sh->ents)[RND(sh->nents)];
	    dbg(CON, ("connecting to '%s' %s:%i from %d.%d.%d.%d ... ",
		au->proxy ? au->proxy->host->name : au->url->host->name, 
		inet_ntoa(addr), port, 
		baddr.s_addr & 255, (baddr.s_addr >> 8) & 255, 
		(baddr.s_addr >> 16) & 255, baddr.s_addr >> 24));
	} else {
	    baddr.s_addr = 0;
	    dbg(CON, ("connecting to '%s' %s:%i ... ",
		au->proxy ? au->proxy->host->name : au->url->host->name, 
		inet_ntoa(addr), port));
	}
	if ((au->socket = tcp_connect(addr, port, baddr)) < 0) {
	    dbge(CON, ("failed!\n"));
	    _deactivate_url(au);
	    if (!num_urls_active)
		die(1, "tcp_connect() keeps failing.");
	    prx(ERR, "tcp_connect() failed!\n");
	    return RT_AGAIN;
	} else {
	    dbge(CON, ("ok\n"));
	    if (au->proxye)
		au->proxye->score++;
	    au->timeout = cur_tv.tv_sec + 
			  au->url->parm->opt->timeout_connect;
	    ls_add(list_urls_request, au);
	    num_urls_active++;
	    if (timerisset(&throttle)) {
		timeradd(&cur_tv, &throttle, next_conn_tv);
		return RT_DONE;
	    }
	    return RT_OK;
	}
}

static void
_disconnect_url(aurl_t *au)
{
    /*  fake -- err: correct our statistics. ;-) */
    if (au->http_done_header && au->size_total)
	total_bytes += -au->size_total + au->size_fetched;

    num_urls_active--;
    if (au->socket != -1)
	close(au->socket);
    if (au->buffer)
	free(au->buffer);
    cq_consume(au->buff, buffe_t, be, {
	cq_rm1st(au->buff);
	free(be);
    });
    if (au->headers)
	free(au->headers);
    if (au->f != -1)
	close(au->f);
}

void
disconnect_url(aurl_t *au)
{
    _disconnect_url(au);
    init_aurl(au);
}

void
deactivate_url(aurl_t *au)
{
    _disconnect_url(au);
    _deactivate_url(au);
}
