/*
 * Copyright (c) 2015 Svyatoslav Mishyn <juef@openmailbox.org>
 *
 * Permission to use, copy, modify, and/or distribute this software for
 * any purpose with or without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stddef.h>
#include <string.h>
#include <tcl.h>

#include "extern/markdown.h"
#include "extern/renderers.h"

#define INPUT_UNIT	 1024
#define OUTPUT_UNIT	 64

#define NS		"soldout"
#define VERSION		"0.1.1"

static	int	 convertMkd(Tcl_Interp *, Tcl_Obj *,
		     const struct mkd_renderer *);
static	int	 x_html(Tcl_Interp *, int, Tcl_Obj *const [], int);

static	int	 escape_html(ClientData, Tcl_Interp *, int, Tcl_Obj *const []);
static	int	 mkd2html(ClientData, Tcl_Interp *, int, Tcl_Obj *const []);
static	int	 mkd2xhtml(ClientData, Tcl_Interp *, int, Tcl_Obj *const []);

int		 Tclsoldout_Init(Tcl_Interp *);

static int
convertMkd(Tcl_Interp *interp, Tcl_Obj *mkd, const struct mkd_renderer *rndr)
{
	struct buf	*ib, *ob;
	size_t		 len;

	len = (size_t)mkd->length;

	if (!len) {
		Tcl_SetObjResult(interp, Tcl_NewStringObj("empty string", 12));
		return TCL_ERROR;
	}

	ib = bufnew(INPUT_UNIT);
	ib->size = len;

	bufgrow(ib, ib->size + INPUT_UNIT);
	memcpy(ib->data, Tcl_GetString(mkd), ib->size + 1);

	ob = bufnew(OUTPUT_UNIT);
	markdown(ob, ib, rndr);

	Tcl_SetObjResult(interp, Tcl_NewStringObj(ob->data, ob->size - 1));
	bufrelease(ib);
	bufrelease(ob);

	return TCL_OK;
}

static int
x_html(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[], int html)
{
	Tcl_Obj				*mkd;
	const struct mkd_renderer	*rndr;
	const char			*opt;

	if (objc == 2) {
		rndr = html ? &mkd_html : &mkd_xhtml;
		mkd = objv[1];

	} else if (objc == 3 && objv[1]->bytes[0] == '-') {
		opt = Tcl_GetString(objv[1]);

		if (strcmp(opt, "-discount") == 0) {
			rndr = html ? &discount_html : &discount_xhtml;

		} else if (strcmp(opt, "-natext") == 0) {
			rndr = html ? &nat_html : &nat_xhtml;

		} else {
			Tcl_SetObjResult(interp, Tcl_ObjPrintf(
			    "unknown option \"%s\", must be: "
			    "-discount or -natext", opt));
			return TCL_ERROR;
		}

		mkd = objv[2];

	} else {
		Tcl_WrongNumArgs(interp, 1, objv,
		    "?-discount|-natext? markdown");
		return TCL_ERROR;
	}

	if (convertMkd(interp, mkd, rndr) != TCL_OK) {
		return TCL_ERROR;
	}

	return TCL_OK;
}

/* only '<', '>', '&' and '"' */
static int
escape_html(ClientData clientData, Tcl_Interp *interp, int objc,
    Tcl_Obj *const objv[])
{
	struct buf	*ob;
	size_t		 len;

	if (objc == 2) {
		len = (size_t)objv[1]->length;

		if (!len) {
			Tcl_SetObjResult(interp,
			    Tcl_NewStringObj("empty string", 12));
			return TCL_ERROR;
		}

		ob = bufnew(OUTPUT_UNIT);
		lus_attr_escape(ob, Tcl_GetString(objv[1]), len);

		Tcl_SetObjResult(interp, Tcl_NewStringObj(ob->data, ob->size));
		bufrelease(ob);

		return TCL_OK;

	} else {
		Tcl_WrongNumArgs(interp, 1, objv, "string");
		return TCL_ERROR;
	}
}

static int
mkd2html(ClientData clientData, Tcl_Interp *interp, int objc,
    Tcl_Obj *const objv[])
{
	return x_html(interp, objc, objv, 1);
}

static int
mkd2xhtml(ClientData clientData, Tcl_Interp *interp, int objc,
    Tcl_Obj *const objv[])
{
	return x_html(interp, objc, objv, 0);
}

int
Tclsoldout_Init(Tcl_Interp *interp)
{
	Tcl_Namespace	*nsPtr;

	if (
#ifdef USE_TCL_STUBS
	    Tcl_InitStubs(interp, "8.1", 0)
#else
	    Tcl_PkgRequire(interp, "Tcl", "8.1", 0)
#endif
	    == NULL) {
		return TCL_ERROR;
	}

	if (Tcl_PkgProvide(interp, "TclSoldout", VERSION) != TCL_OK) {
		return TCL_ERROR;
	}

	if ((nsPtr = Tcl_CreateNamespace(interp, NS, NULL, NULL)) == NULL) {
		return TCL_ERROR;
	}

	if (Tcl_Export(interp, nsPtr, "*", 0) != TCL_OK) {
		return TCL_ERROR;
	}

	Tcl_CreateObjCommand(interp, NS "::escape_html", escape_html, NULL,
	    NULL);
	Tcl_CreateObjCommand(interp, NS "::mkd2html", mkd2html, NULL, NULL);
	Tcl_CreateObjCommand(interp, NS "::mkd2xhtml", mkd2xhtml, NULL, NULL);

	return TCL_OK;
}
