/*
 *
 *  * Copyright (C) 2023, KylinSoft Co., Ltd.
 *  *
 *  * This program is free software: you can redistribute it and/or modify
 *  * it under the terms of the GNU General Public License as published by
 *  * the Free Software Foundation, either version 3 of the License, or
 *  * (at your option) any later version.
 *  *
 *  * This program is distributed in the hope that it will be useful,
 *  * but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  * GNU General Public License for more details.
 *  *
 *  * You should have received a copy of the GNU General Public License
 *  * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *  *
 *  * Authors:  Zhang <zhangyuanyuan1@kylinos.cn>
 *
 */

import QtQuick 2.12
import QtQuick.Layouts 1.12
import QtQml 2.12
import QtQuick.Window 2.12
import QtQuick.Controls 2.5
import org.ukui.panel.items 1.0
import org.ukui.windowThumbnail 1.0

Rectangle {
    id: root
    anchors.centerIn: parent
    radius: wThumbnailView.radius
    width: listview.width
    height: listview.height
    color: "transparent"

    MouseArea {
        id: rootMouseArea
        anchors.fill: parent
        hoverEnabled: true
        onEntered: {
            wThumbnailView.viewVisible = true;
        }

        onExited: {
            wThumbnailView.viewVisible = false;
        }

        Component {
            id: listPage
            StyleBackground {
                id: windowView
                readonly property bool isList: wThumbnailView.isShowByList
                useStyleTransparency: true
                paletteRole: PaletteRole.Window
                alpha: 1.0;
                borderColor: PaletteRole.BrightText
                borderAlpha: 0.15
                border.width: 1.0;
                width: isList ? 220 : wThumbnailView.windowSize(index).width
                height: isList ? 40 : wThumbnailView.windowSize(index).height
                radius: wThumbnailView.radius

                Component.onCompleted: {
                    if (isList) {
                        windowView.width = 220;
                    } else {
                        if (windowThumbnail.thumbnailMprisModel.count > 0) {
                            windowView.width = Math.max(windowView.width, 148);
                            windowView.height = Math.max(windowView.height, 192);
                            wThumbnailView.updateMprisWindowSize(index, windowView.width, windowView.height);
                        }
                    }
                }

                Timer {
                    id: timer
                    interval : 400
                    repeat : false
                    onTriggered: {
                        if (windowViewMouseArea.containsMouse) {
                            timer.stop();
                            wThumbnailView.sendSigToKwin(modelData);
                        }
                    }
                }

                MouseArea {
                    id: windowViewMouseArea
                    anchors.fill: parent
                    hoverEnabled: true
                    acceptedButtons: Qt.LeftButton | Qt.RightButton

                    onEntered: {
                        closeButton.visible = true;
                        //高亮显示
                        windowView.paletteRole = PaletteRole.Button

                        timer.restart();
                    }

                    onExited: {
                        closeButton.visible = false;
                        //非高亮显示
                        windowView.paletteRole = PaletteRole.Window
                    }

                    onClicked: {
                        windowView.paletteRole = PaletteRole.Button;
                        if (mouse.button === Qt.RightButton) {
                            wThumbnailView.openMenu(modelData);
                        } else {
                            thumbnailModel.activateWindow(modelData);
                            wThumbnailView.viewVisible = false;
                        }
                        windowView.paletteRole = PaletteRole.Window
                        closeButton.visible = false;
                    }

                    ColumnLayout {
                        anchors.fill: parent
                        anchors.leftMargin: 8
                        anchors.rightMargin: 8
                        anchors.topMargin: 8
                        anchors.bottomMargin: 8
                        spacing: 8

                        RowLayout {
                            Layout.fillWidth: true
                            Layout.preferredHeight: 32
                            Layout.maximumHeight: 32

                            ThemeIcon {
                                Layout.preferredHeight: windowView.isList ? 16 : 24
                                Layout.preferredWidth: windowView.isList ? 16 : 24
                                source: thumbnailModel.getWindowIcon(modelData)
                            }
                            StyleText {
                                id: title
                                Layout.fillHeight: true
                                Layout.fillWidth: true
                                text: thumbnailModel.getWindowTitle(modelData)
                                verticalAlignment: Text.AlignVCenter
                                elide: Text.ElideRight
                                wrapMode: Text.Wrap
                                maximumLineCount: 1
                                StyleToolTip.text: thumbnailModel.getWindowTitle(modelData)

                                MouseArea {
                                    anchors.fill: parent
                                    hoverEnabled: true
                                    propagateComposedEvents: true
                                    onEntered: {
                                        title.StyleToolTip.show(title.mapToGlobal(mouseX, mouseY));
                                    }
                                    onExited: {
                                        title.StyleToolTip.hide();
                                    }
                                }
                            }
                            //关闭按钮
                            Rectangle {
                                id: closeButton
                                Layout.preferredWidth: 18
                                Layout.preferredHeight: 18
                                radius: wThumbnailView.radius < 6 ? wThumbnailView.radius : 6
                                Layout.alignment: Qt.AlignRight
                                visible: false
                                color: closeBtnArea.containsPress ? "firebrick" :
                                           (closeBtnArea.containsMouse ? "red" : "transparent")

                                ThemeIcon {
                                    id: icon
                                    width: 16
                                    height: width
                                    anchors.centerIn: parent
                                    highLight: false
                                    source: "window-close-symbolic"
                                }

                                MouseArea {
                                    id: closeBtnArea
                                    anchors.fill: parent
                                    hoverEnabled: true
                                    propagateComposedEvents: true
                                    onEntered: {
                                        icon.highLight = true;
                                    }

                                    onExited: {
                                        icon.highLight = false;
                                    }
                                    onClicked: {
                                        icon.highLight = true;
                                        thumbnailModel.closeWindow(modelData);
                                    }
                                }
                            }
                        } //RowLayout
                        WindowThumbnail {
                            id: windowThumbnail
                            Layout.fillWidth: true
                            Layout.fillHeight: true
                            winId: windowView.isList ? "" : Number.parseInt(modelData)
                            visible: !windowView.isList
                        }
                    } //ColumnLayout
                }
            }
        } //Component

        ListView {
            id:listview
            anchors.left: parent.left
            width: childrenRect.width
            height: childrenRect.height
            interactive: false
            orientation: ListView.Horizontal
            spacing: 8

            delegate: listPage
            model: wThumbnailView.viewModel
            boundsBehavior: Flickable.StopAtBounds
            ScrollBar.vertical: listViewScrollBar

            onChildrenRectChanged: {
                    wThumbnailView.width = root.width
            }

            onModelChanged: {
                wThumbnailView.calculateWindowsSize(model);
                wThumbnailView.updateViewMode();
                listview.state = wThumbnailView.isShowByList ?
                            "VerticalList" : (wThumbnailView. isShowHorizontalView ? "HorizontalPage": "VerticalPage")
            }

            states: [
                State {
                    name: "HorizontalPage"
                    PropertyChanges { target: listview; orientation: ListView.Horizontal;
                        interactive: false;
                        width: childrenRect.width; height: 192 }
                    PropertyChanges { target: root; width: listview.width }
                },
                State {
                    name: "VerticalPage"
                    PropertyChanges { target: listview; orientation: ListView.Vertical;
                        interactive: false;
                        width: 272; height: childrenRect.height }
                    PropertyChanges { target: root; width: listview.width }
                },
                State {
                    name: "VerticalList"
                    PropertyChanges { target: listview; orientation: ListView.Vertical;
                        interactive: true;
                        width: 220; height: Math.min(childrenRect.height, Screen.desktopAvailableHeight - 16) }
                    PropertyChanges { target: root; width: 220 + 14 }
                }
            ]
        }

        StyleScrollBar {
            id: listViewScrollBar
            anchors.left: listview.right
            anchors.top: listview.top
            height: listview.height
            width: 14
            visual: rootMouseArea.containsMouse
        }

    }
}
